/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Utils;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FileDialog;
import java.awt.FlowLayout;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Vector;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import TreeSnatcher.Core.Branch;
import TreeSnatcher.Core.ImageOperations;
import TreeSnatcher.Core.MainWindow;
import TreeSnatcher.Core.Tip;
import TreeSnatcher.Core.TreeTopology;
import TreeSnatcher.GUI.ImageBuffer;
import TreeSnatcher.GUI.ImagePanel;
import TreeSnatcher.GUI.InfiniteProgressPanel;
import TreeSnatcher.GUI.StatusBar;
import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.Core.TreeNode;

public class FileOperations {
	public MainWindow owner;
	private InfiniteProgressPanel progressPane;
	private ImageBuffer imageBuffer;
	private StatusBar statusBar;
	private Wizard wizard;
	private TreeTopology topology;
	private ImagePanel imagePanel;
	private ImageOperations imageOperations;
	private Dimension screen;
	protected int width = 0;
	protected int height = 0;

	public FileOperations(MainWindow owner, InfiniteProgressPanel progressPane,
			ImageBuffer imageBuffer, Wizard wizard, StatusBar statusBar) {
		this.owner = owner;
		this.progressPane = progressPane;
		this.imageBuffer = imageBuffer;
		this.wizard = wizard;
		this.statusBar = statusBar;

		// Get the dimension of the Desktop
		screen = Toolkit.getDefaultToolkit().getScreenSize();
	}

	public BufferedImage showOpenFileDialog() {
		BufferedImage image = null;
		JFileChooser jfc = new JFileChooser();
		jfc.addChoosableFileFilter(new ImageFilter());
		jfc.setDialogTitle("Open an image file");
		jfc.setCurrentDirectory(new File(Preferences.readImgPath));
		jfc.setFileSelectionMode(JFileChooser.FILES_ONLY);
		ImageFileChooser preview = new ImageFileChooser(jfc);
		jfc.addPropertyChangeListener(preview);
		jfc.setAccessory(preview);

		int option = jfc.showOpenDialog(null);

		if (option == JFileChooser.APPROVE_OPTION) {
			progressPane.start();
			File file = jfc.getSelectedFile();
			String path = file.getPath();
			image = ImageLoader.getImage(path);
			wizard.blankImage = false;
			wizard.imageLoaded = true;
			wizard.imagePath = path;

			// Store the directory from which the file was loaded in the
			// Preferences file
			// First cut the file name
			Preferences.readImgPath = path;

			if (!Preferences.writePrefs())
				System.out.println("Could not write the Preferences text file");
		}
		return image;
	}

	// Open a snapshot restores the original image, the binarized image and the
	// intermediate tree topology
	public void showOpenSnapshotDialog() {
		Point viewPosition = null;
		String path = null;
		Vector<Branch> branches = null;
		Vector<TreeNode> nodes = null;
		BufferedImage image = null;
		JFileChooser jfc = new JFileChooser();
		jfc.addChoosableFileFilter(new SnapshotFilter());
		jfc.setDialogTitle("Restore from Snapshot");
		jfc.setCurrentDirectory(new File(Preferences.readSnapshotPath));
		int option = jfc.showOpenDialog(null);
		if (option == JFileChooser.APPROVE_OPTION) {
			imagePanel.setVisible(false);

			File file = jfc.getSelectedFile();
			path = file.getPath();

			// Load the topology data
			try {
				FileInputStream fs = new FileInputStream(file);
				ObjectInputStream is = new ObjectInputStream(fs);
				// Restore topology objects
				branches = (Vector<Branch>) is.readObject();
				nodes = (Vector<TreeNode>) is.readObject();
				TreeNode.num = (Integer) is.readObject();
				TreeNode.signalNode = (Integer) is.readObject();
				Branch.num = (Integer) is.readObject();

				Tip.nameCnt = (Integer) is.readObject();

				// Restore the scrollpane's viewport position
				viewPosition = (Point) is.readObject();

				// Restore wizard flags
				wizard.useWizard = (Boolean) is.readObject();
				wizard.markFloodSeed = (Boolean) is.readObject();
				wizard.smallMarks = (Boolean) is.readObject();
				wizard.imageLoaded = (Boolean) is.readObject();
				wizard.isBinarized = (Boolean) is.readObject();
				wizard.isSkeletonized = (Boolean) is.readObject();
				wizard.autoSkeletonize = (Boolean) is.readObject();
				wizard.autoFlood = (Boolean) is.readObject();
				wizard.autoClean = (Boolean) is.readObject();
				wizard.showSourceImageInMagnifier = (Boolean) is.readObject();
				wizard.imageCropped = (Boolean) is.readObject();
				wizard.floodedRegionExists = (Boolean) is.readObject();
				wizard.highlightFGColorInMagnifier = (Boolean) is.readObject();
				wizard.floodFillPerformed = (Boolean) is.readObject();
				wizard.useFancyCursors = (Boolean) is.readObject();
				wizard.useOneNodeMode = (Boolean) is.readObject();
				wizard.showAllBranches = (Boolean) is.readObject();
				wizard.showTracedPaths = (Boolean) is.readObject();
				wizard.suppressTinyBranches = (Boolean) is.readObject();
				wizard.nodesStickToTree = (Boolean) is.readObject();
				wizard.floodedAreaExtracted = (Boolean) is.readObject();
				wizard.mayDiscardUserGeneratedObjects = (Boolean) is
						.readObject();
				wizard.blankImage = (Boolean) is.readObject();
				wizard.drawBranchLength = (Boolean) is.readObject();
				wizard.drawTaxonName = (Boolean) is.readObject();
				wizard.highlightBranchesWithoutLength = (Boolean) is
						.readObject();
				wizard.highlightTipsWithoutName = (Boolean) is.readObject();
				wizard.showResultFrame = (Boolean) is.readObject();
				wizard.treeIsBifurcating = (Boolean) is.readObject();
				wizard.nodesCollected = (Boolean) is.readObject();

				wizard.selectionTrimmedToBoundaries = (Boolean) is.readObject();

				wizard.topologyDeterminationFinished = (Boolean) is
						.readObject();
				wizard.colReplace = (Boolean) is.readObject();
				wizard.colKeep = (Boolean) is.readObject();
				wizard.colExactColor = (Boolean) is.readObject();
				wizard.colSelectionColors = (Boolean) is.readObject();
				wizard.colNumSelectionColors = (Integer) is.readObject();

				wizard.transparencyRatio = (Float) is.readObject();
				wizard.fgColorMagnifier = (Integer) is.readObject();
				wizard.treeFloodColor = (Integer) is.readObject();
				wizard.lastColorPicked = (Integer) is.readObject();
				wizard.fromColor = (Integer) is.readObject();
				wizard.toColor = (Integer) is.readObject();
				wizard.localFillColor = (Integer) is.readObject();
				wizard.treePixels = (Integer) is.readObject();
				Wizard.scaleBarLengthInPixels = (Integer) is.readObject();
				wizard.referenceNodeId = (Integer) is.readObject();
				wizard.colorDistance = (Integer) is.readObject();
				wizard.quantNumColors = (Integer) is.readObject();
				wizard.mostRecentTopologyError = (Integer) is.readObject();
				wizard.tinyBranchLength = (Integer) is.readObject();
				Wizard.manualScaleBarLength = (Double) is.readObject();
				wizard.floodSeed = (Point) is.readObject();
				wizard.latestTreeLocation = (Point) is.readObject();
				wizard.floodedPixels = (Integer) is.readObject();
				Wizard.useUserAssignedLengths = (Boolean) is.readObject();
				Wizard.useCalculatedLengths = (Boolean) is.readObject();
				Wizard.useMixedLengths = (Boolean) is.readObject();
				wizard.ratioSliderEnabled = (Boolean) is.readObject();
				wizard.colorNewickString = (Boolean) is.readObject();
				wizard.ratioSliderValue = (Integer) is.readObject();
				wizard.includeBranchLengths = (Boolean) is.readObject();
				wizard.drawingStrokeWidth = (Integer) is.readObject();
				wizard.eraserStrokeWidth = (Integer) is.readObject();
				wizard.topologyType = (Integer) is.readObject();
				wizard.useElaboratePathFinding = (Boolean) is.readObject();
				wizard.lookaheadDistance = (Integer) is.readObject();

				is.close();

				Wizard.undoHistoryEmptyAfterLastSave = true;

			} catch (ClassNotFoundException e) {
			} catch (IOException e) {
			} catch (IllegalArgumentException e) {
			}

			// Load the image(s)
			wizard.imagePath = path;
			path = path.substring(0, path.length() - 4); // Cut ".ser"
			String newPath = path + "_c.PNG";
			image = ImageLoader.getImage(newPath);

			if (image != null) {
				image = imageOperations.addBorderToImage(image);
				imagePanel.issueSnapshot(image);
			}

			newPath = path + "_b.PNG";
			image = ImageLoader.getImage(newPath);
			if (image != null)
				imageBuffer.storeBinarizedImage(image);

			newPath = path + "_o.PNG";
			image = ImageLoader.getImage(newPath);
			if (image != null) {
				imageBuffer.storeSourceImage(image);
			}

			wizard.blankImage = false;
			wizard.imageLoaded = true;

			// Store the directory from which the file was loaded in the
			// Preferences file
			Preferences.readSnapshotPath = jfc.getCurrentDirectory().toString();
			if (!Preferences.writePrefs())
				System.out.println("Could not write the Preferences text file");
			topology.issueSnapshot(branches, nodes);

			// Determine the number of fractional digits needed for all branch
			// lengths
			NumberUtility.setNumFractionDigits(branches);
			imagePanel.updateResultFrame();
			imagePanel.setViewportPosition(viewPosition);
			imagePanel.setVisible(true);
		}
	}

	public BufferedImage showNewFileDialog() {
		final JSpinner widthSpinner = new JSpinner();
		final JSpinner heightSpinner = new JSpinner();
		widthSpinner.setMaximumSize(new Dimension(30, 20));
		heightSpinner.setMaximumSize(new Dimension(30, 20));
		SpinnerNumberModel model1 = new SpinnerNumberModel(1, 1, 1000, 1);
		SpinnerNumberModel model2 = new SpinnerNumberModel(1, 1, 1000, 1);
		widthSpinner.setModel(model1);
		widthSpinner.setValue(400);
		width = 400;
		heightSpinner.setModel(model2);
		heightSpinner.setValue(400);
		height = 400;
		widthSpinner.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				width = (Integer) ((SpinnerNumberModel) widthSpinner.getModel())
						.getValue();
			}
		});
		heightSpinner.addChangeListener(new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				height = (Integer) ((SpinnerNumberModel) heightSpinner
						.getModel()).getValue();
			}
		});

		Object[] message = { "Width/Pixels", widthSpinner, "Height/Pixels",
				heightSpinner };
		JOptionPane pane = new JOptionPane(message, JOptionPane.PLAIN_MESSAGE,
				JOptionPane.OK_CANCEL_OPTION);
		pane.setLayout(new FlowLayout());
		pane.setPreferredSize(new Dimension(280, 180));
		pane.createDialog(null, "Create New Image").setVisible(true);

		if ((Integer) (pane.getValue()) == JOptionPane.OK_OPTION) {
			if ((width >= 50) && (width <= 1000) && (height >= 50)
					&& (height <= 1000)) {
				BufferedImage image = new BufferedImage(width, height,
						BufferedImage.TYPE_INT_ARGB);
				Graphics2D g2 = (Graphics2D) image.getGraphics();
				g2.setColor(Color.white);
				g2.fillRect(0, 0, width, height);
				wizard.reset();
				wizard.blankImage = true;
				wizard.imageLoaded = false;

				Wizard.undoHistoryEmptyAfterLastSave = true;

				return image;
			} else
				statusBar
						.showMessage(
								"Only image sizes between 50x50 and 1000x1000 pixels are allowed",
								null, null);
		}
		return null;
	}

	public void showSaveIllustrationDialog(boolean[] params) {
		// AWT version of a file dialog is more authentic
		boolean mayWriteFile = true;
		BufferedImage image = null, overlayImage = null;
		String fileName = "";

		FileDialog fileSaver = new FileDialog(owner,
				"Save Image with layers as PNG", FileDialog.SAVE);
		fileSaver.setDirectory(Preferences.writeImgPath);
		fileSaver.setFilenameFilter(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.toLowerCase().endsWith(".png");
			}
		});

		fileSaver.setLocation(
				(int) (screen.getWidth() - fileSaver.getWidth()) / 2,
				(int) (screen.getHeight() - fileSaver.getHeight()) / 2);
		fileSaver.setVisible(true);
		if (!fileSaver.getFile().endsWith(".png"))
			fileName = fileSaver.getFile() + ".png";
		else
			fileName = fileSaver.getFile();
		String path = fileSaver.getDirectory()
				+ System.getProperty("file.separator") + fileName;

		File file = new File(path);
		if (!fileSaver.getFile().endsWith(".png") && file.exists()) {
			Object[] options = { "Yes, overwrite", "No, do not overwrite" };

			int answer = JOptionPane.showOptionDialog(owner, "The file "
					+ fileName + " exists. Overwrite it?", path,
					JOptionPane.YES_NO_CANCEL_OPTION,
					JOptionPane.QUESTION_MESSAGE, null, options, options[0]);

			if (answer == JOptionPane.YES_OPTION)
				mayWriteFile = true;
			else
				mayWriteFile = false;
		}

		if (mayWriteFile) {
			overlayImage = imagePanel.getOverlayImage(params);
			image = imageBuffer.getBorderlessImage(overlayImage);
			if (fileSaver.getFile() != null)
				new ImageSaver(image, file);

			// Store the directory from which the file was saved in the
			// Preferences file
			Preferences.writeImgPath = fileSaver.getDirectory();
			if (!Preferences.writePrefs())
				System.out.println("Could not write the Preferences text file");
		}
	}

	public void showSaveNewickDialog(String newickExpression) {
		// Abort if the newick expression has not been calculated yet, otherwise
		// open a save dialog
		if ((newickExpression == null) || (newickExpression.equals(""))) {

			statusBar
					.showMessage(
							"Nothing to save as Newick expression is empty",
							null, null);
		} else {
			FileDialog fileSaver = new FileDialog(owner,
					"Save Newick Expression as Text", FileDialog.SAVE);
			fileSaver.setDirectory(Preferences.writeNewickPath);
			fileSaver.setLocation((int) (screen.getWidth() - fileSaver
					.getWidth()) / 2, (int) (screen.getHeight() - fileSaver
					.getHeight()) / 2);
			fileSaver.setVisible(true);
			String path = fileSaver.getDirectory()
					+ System.getProperty("file.separator")
					+ fileSaver.getFile();

			if (fileSaver.getFile() != null) {
				File file = new File(path);

				try {
					FileWriter fileWriter = new FileWriter(file);
					fileWriter.write(newickExpression);
					fileWriter.close();
				} catch (IOException e) {
					JOptionPane.showMessageDialog(null,
							"The file could not be saved.");
				}

				// Store the directory from which the file was saved in the
				// Preferences file
				Preferences.writeNewickPath = fileSaver.getDirectory();
				if (!Preferences.writePrefs())
					System.out
							.println("Could not write the Preferences text file");
			}
		}
	}

	// Stores the current state of work as multiple files: image, binarized
	// image, nodes and branches
	public void showSaveSnapshotDialog() {
		boolean mayWriteFile = true;
		ObjectOutputStream os = null;
		File file = null;
		FileOutputStream fs = null;
		FileDialog fileSaver = new FileDialog(owner, "Save a Snapshot",
				FileDialog.SAVE);
		fileSaver.setDirectory(Preferences.writeSnapshotPath);
		fileSaver.setLocation(
				(int) (screen.getWidth() - fileSaver.getWidth()) / 2,
				(int) (screen.getHeight() - fileSaver.getHeight()) / 2);
		fileSaver.setVisible(true);
		fileSaver.setFilenameFilter(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.toLowerCase().endsWith(".ser");
			}
		});

		String path = fileSaver.getDirectory() + fileSaver.getFile();
		if (!path.toLowerCase().endsWith(".ser"))
			path = path + ".ser";

		if (path != null) {
			// Store objects
			file = new File(path);

			if (!fileSaver.getFile().endsWith(".ser") && file.exists()) {
				Object[] options = { "Yes, overwrite", "No, do not overwrite" };

				int answer = JOptionPane
						.showOptionDialog(owner, "The file " + file
								+ " exists. Overwrite it?", path,
								JOptionPane.YES_NO_CANCEL_OPTION,
								JOptionPane.QUESTION_MESSAGE, null, options,
								options[0]);

				if (answer == JOptionPane.YES_OPTION)
					mayWriteFile = true;
				else
					mayWriteFile = false;
			}
		}

		if (mayWriteFile) {
			try {
				fs = new FileOutputStream(file);
				os = new ObjectOutputStream(fs);

				// Store branches
				os.writeObject(topology.getBranches());
				// Store nodes
				os.writeObject(topology.getNodes());
				// Store static objects
				os.writeObject(TreeNode.num);
				os.writeObject(TreeNode.signalNode);
				os.writeObject(Branch.num);

				os.writeObject(Tip.nameCnt);

				Point viewPosition = imagePanel.getViewportPosition();
				os.writeObject(viewPosition);

				// Store wizard flags
				os.writeObject(wizard.useWizard);
				os.writeObject(wizard.markFloodSeed);
				os.writeObject(wizard.smallMarks);
				os.writeObject(wizard.imageLoaded);
				os.writeObject(wizard.isBinarized);
				os.writeObject(wizard.isSkeletonized);
				os.writeObject(wizard.autoSkeletonize);
				os.writeObject(wizard.autoFlood);
				os.writeObject(wizard.autoClean);
				os.writeObject(wizard.showSourceImageInMagnifier);
				os.writeObject(wizard.imageCropped);
				os.writeObject(wizard.floodedRegionExists);
				os.writeObject(wizard.highlightFGColorInMagnifier);
				os.writeObject(wizard.floodFillPerformed);
				os.writeObject(wizard.useFancyCursors);
				os.writeObject(wizard.useOneNodeMode);
				os.writeObject(wizard.showAllBranches);
				os.writeObject(wizard.showTracedPaths);
				os.writeObject(wizard.suppressTinyBranches);
				os.writeObject(wizard.nodesStickToTree);
				os.writeObject(wizard.floodedAreaExtracted);
				os.writeObject(wizard.mayDiscardUserGeneratedObjects);
				os.writeObject(wizard.blankImage);
				os.writeObject(wizard.drawBranchLength);
				os.writeObject(wizard.drawTaxonName);
				os.writeObject(wizard.highlightBranchesWithoutLength);
				os.writeObject(wizard.highlightTipsWithoutName);
				os.writeObject(wizard.showResultFrame);
				os.writeObject(wizard.treeIsBifurcating);
				os.writeObject(wizard.nodesCollected);
				os.writeObject(wizard.selectionTrimmedToBoundaries);
				os.writeObject(wizard.topologyDeterminationFinished);
				os.writeObject(wizard.colReplace);
				os.writeObject(wizard.colKeep);
				os.writeObject(wizard.colExactColor);
				os.writeObject(wizard.colSelectionColors);
				os.writeObject(wizard.colNumSelectionColors);
				os.writeObject(wizard.transparencyRatio);
				os.writeObject(wizard.fgColorMagnifier);
				os.writeObject(wizard.treeFloodColor);
				os.writeObject(wizard.lastColorPicked);
				os.writeObject(wizard.fromColor);
				os.writeObject(wizard.toColor);
				os.writeObject(wizard.localFillColor);
				os.writeObject(wizard.treePixels);
				os.writeObject(Wizard.scaleBarLengthInPixels);
				os.writeObject(wizard.referenceNodeId);
				os.writeObject(wizard.colorDistance);
				os.writeObject(wizard.quantNumColors);
				os.writeObject(wizard.mostRecentTopologyError);
				os.writeObject(wizard.tinyBranchLength);
				os.writeObject(Wizard.manualScaleBarLength);
				os.writeObject(wizard.floodSeed);
				os.writeObject(wizard.latestTreeLocation);
				os.writeObject(wizard.floodedPixels);
				os.writeObject(Wizard.useUserAssignedLengths);
				os.writeObject(Wizard.useCalculatedLengths);
				os.writeObject(Wizard.useMixedLengths);
				os.writeObject(wizard.ratioSliderEnabled);
				os.writeObject(wizard.colorNewickString);
				os.writeObject(wizard.ratioSliderValue);
				os.writeObject(wizard.includeBranchLengths);
				os.writeObject(wizard.drawingStrokeWidth);
				os.writeObject(wizard.eraserStrokeWidth);
				os.writeObject(wizard.topologyType);
				os.writeObject(wizard.useElaboratePathFinding);
				os.writeObject(wizard.lookaheadDistance);

				Wizard.undoHistoryEmptyAfterLastSave = true;
				wizard.imagePath = path;

				os.close();
			} catch (IOException e) {
				System.out.println("Could not write Branches");
			}

			// Save binarized image
			path = path.substring(0, path.length() - 4);
			String newPath = path + "_b.PNG";
			file = new File(newPath);
			if (imageBuffer.getBinarizedImage() != null) {
				BufferedImage binaryImage = imageBuffer.getBinarizedImage();
				new ImageSaver(binaryImage, file);
			}

			// Save current image
			newPath = path + "_c.PNG";
			file = new File(newPath);
			if (imageBuffer.getBorderlessImage() != null) {
				new ImageSaver(imageBuffer.getBorderlessImage(), file);
			}

			// Save original image
			newPath = path + "_o.PNG";
			file = new File(newPath);
			if (imageBuffer.getSourceImage() != null) {
				BufferedImage sourceImage = imageBuffer.getSourceImage();
				new ImageSaver(sourceImage, file);
			}

			// Store the directory from which the file was saved in the
			// Preferences file
			Preferences.writeSnapshotPath = fileSaver.getDirectory();
			if (!Preferences.writePrefs())
				System.out.println("Could not write the Preferences text file");
		}
	}

	// Stores an image portion on harddisk
	public void saveProtocolImage(String name, BufferedImage portion) {
		File file = new File(Preferences.protocolPath + File.separator + name
				+ ".png");
		new ImageSaver(portion, file);
	}

	// Copies an object via serialization
	// From: Guido Krueger, Handbuch der JAVA-Programmierung, Verlag
	// Addison-Wesley
	public static Object seriaCopy(Object o) throws IOException,
			ClassNotFoundException {
		// Serialization of the object
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ObjectOutputStream os = new ObjectOutputStream(out);
		os.writeObject(o);
		os.flush();
		// Deserialization of the object
		ByteArrayInputStream in = new ByteArrayInputStream(out.toByteArray());
		ObjectInputStream is = new ObjectInputStream(in);
		Object ret = is.readObject();
		is.close();
		os.close();
		return ret;
	}

	public void setObjectReferences(TreeTopology tt, ImagePanel ip,
			ImageOperations io) {
		this.topology = tt;
		this.imagePanel = ip;
		this.imageOperations = io;
	}
}
