/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.GUI;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.HeadlessException;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import TreeSnatcher.Core.Constants;
import TreeSnatcher.Core.MainWindow;
import TreeSnatcher.Core.TreeTopology;
import TreeSnatcher.Utils.FileOperations;
import TreeSnatcher.GUI.ColorPane;
import TreeSnatcher.GUI.Wizard;

public class ResultFrame extends JPanel implements Constants, ActionListener {
	private static final long serialVersionUID = 1L;
	private JScrollPane textScrollPane;
	private TreeTopology topology;
	private FileOperations fileOp;
	private String defaultText = "The Newick expression will appear here.";
	private Clipboard systemClipboard;
	private ColorPane ctp;
	private Wizard wizard;
	private Color colLengths, col;

	public ResultFrame(MainWindow owner, JScrollPane scrollPane) {
		this.setBorder(BorderFactory.createLoweredBevelBorder());
		this.setPreferredSize(new Dimension(800, 180));
		this.setMinimumSize(this.getPreferredSize());
		this.setMaximumSize(this.getPreferredSize());
		this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));

		ctp = new ColorPane();
		ctp.setBorder(BorderFactory.createRaisedBevelBorder());

		JPanel textPanel = new JPanel(new BorderLayout());
		JPanel buttonPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT, 4, 2));
		textScrollPane = new JScrollPane(ctp);
		textScrollPane.setPreferredSize(new Dimension(800, 144));
		textPanel.add(textScrollPane, BorderLayout.NORTH);
		this.setBorder(BorderFactory.createLineBorder(Color.gray, 2));
		this.add(textPanel);
		buttonPanel.setPreferredSize(new Dimension(800, 16));
		JButton copy = new JButton("Copy");
		JButton save = new JButton("Export");
		save.addActionListener(this);
		copy.addActionListener(this);
		copy.setMaximumSize(new Dimension(80, 12));
		save.setMaximumSize(new Dimension(80, 12));
		buttonPanel.add(save);
		buttonPanel.add(copy);
		textPanel.add(buttonPanel, BorderLayout.CENTER);
		setCursor(new Cursor(Cursor.HAND_CURSOR));
		this.invalidate();

		// Gets the singleton instance of the system Clipboard which interfaces
		// with clipboard facilities provided by the native platform.
		// This clipboard enables data transfer between Java programs and native
		// applications which use native clipboard facilities.
		try {
			systemClipboard = Toolkit.getDefaultToolkit().getSystemClipboard();

		} catch (HeadlessException h) {
		}

	}

	public void showNewickString(String newickString) {
		if (newickString != null) {
			ctp.setVisible(false);
			int rows = Math.round(newickString.length() / 104) + 1;
			ctp.setPreferredSize(new Dimension(800, 2 * rows * 20));
			textScrollPane.setViewportView(ctp);
			this.invalidate();
			ctp.setEditable(true);
			ctp.setText("");

			if (wizard.colorNewickString)
				colLengths = defaultNewickLengthsColor;
			else
				colLengths = defaultNewickColor;
			col = defaultNewickColor;

			int i = 0;
			while (i < newickString.length()) {
				String c = newickString.substring(i, i + 1);
				if (c.equals(":")) {
					while ((c.equals(".")) || (c.equals("1"))
							|| (c.equals("2")) || (c.equals("3"))
							|| (c.equals("4")) || (c.equals("5"))
							|| (c.equals("6")) || (c.equals("7"))
							|| (c.equals("8")) || (c.equals("9"))
							|| (c.equals("0")) || (c.equals(":"))) {
						ctp.append(colLengths, c); // Print the branch lengths
						// in a certain color
						i++;
						c = newickString.substring(i, i + 1);
					}
				}
				ctp.append(col, c); // Print everything else in a different
				// shade
				i++;
			}
			ctp.setEditable(false);
			ctp.setVisible(true);
		}
	}

	public Dimension getPreferredSize() {
		return new Dimension(800, 180);
	}

	public void setObjectReferences(GUIActions ga, FileOperations fo,
			TreeTopology tt) {
		this.fileOp = fo;
		this.topology = tt;
	}

	public void reset() {
		ctp.setPreferredSize(new Dimension(800, 180));
		ctp.setText(defaultText);
	}

	public void actionPerformed(ActionEvent e) {
		if (e.getActionCommand().equals("Export")) {
			fileOp.showSaveNewickDialog(topology.getNewickExpression());
		} else if (e.getActionCommand().equals("Copy")) {
			if (systemClipboard != null) {
				StringSelection transferableString = new StringSelection(
						topology.getNewickExpression());
				Transferable t = (Transferable) transferableString;
				ClipboardOwner co = (ClipboardOwner) transferableString;
				if (!transferableString.equals(""))
					systemClipboard.setContents(t, co);
			}
		}
	}

	public void setObjectReferences(Wizard wizard) {
		this.wizard = wizard;
	}
}
