/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.GUI;

import java.awt.*;
import java.util.*;

import javax.swing.*;

import TreeSnatcher.Core.Branch;
import TreeSnatcher.Core.Constants;
import TreeSnatcher.Core.TreeNode;
import TreeSnatcher.Core.TreeTopology;

import java.awt.geom.*;
import java.awt.image.*;

public class Magnifier extends JPanel implements Constants {
	private static final long serialVersionUID = 1L;
	private BufferedImage image;
	private BufferedImage lensImage;
	private BufferedImage lensImageSrc;
	private int factor;
	private int imgX;
	private int imgY;
	private int middleX;
	private int middleY;
	protected int offset = 0;
	private AffineTransform lensTransform;
	private AffineTransformOp lensTransformOp;
	protected Graphics2D g2;
	private Line2D line1, line2;
	private ImageBuffer imageBuffer;
	private Wizard wizard;
	protected TreeTopology topology;
	private ImagePanel imagePanel;
	protected Vector<TreeNode> nodesInMagnifier;
	protected Vector<Branch> branchesInMagnifier;

	public Magnifier(int factor, ImageBuffer imageBuffer, Wizard wizard) {
		super();
		this.imageBuffer = imageBuffer;
		this.wizard = wizard;
		this.factor = factor;
		this.setBorder(BorderFactory.createEtchedBorder());
		this.setMaximumSize(new Dimension(144, 144));
		nodesInMagnifier = new Vector<TreeNode>();
		branchesInMagnifier = new Vector<Branch>();
		lensTransform = AffineTransform.getScaleInstance(factor, factor);
		lensTransformOp = new AffineTransformOp(lensTransform,
				AffineTransformOp.TYPE_NEAREST_NEIGHBOR);
		g2 = (Graphics2D) this.getGraphics();
		middleX = 72;
		middleY = 72;

		line1 = new Line2D.Float(middleX, middleY, middleX + 7, middleY);
		line2 = new Line2D.Float(middleX, middleY, middleX, middleY + 7);
	}

	public void setImage(BufferedImage image) {
		if (image != null) {
			this.image = image;
		}
	}

	public void adjust(int x, int y) // (x, y) is the mouse location on
	// ImagePanel
	{
		if (image != null) {
			this.imgX = x;
			this.imgY = y;
			Shape magnifierView = new Rectangle2D.Float(imgX - 9, imgY - 9, 38,
					38);
			nodesInMagnifier = imagePanel.getNodesInSelection(magnifierView);
			branchesInMagnifier = topology.getBranches();
			this.repaint();
		}
	}

	protected void paintComponent(Graphics g) {
		if (g == null)
			return;
		Graphics2D g2 = (Graphics2D) g;
		g2.setStroke(new BasicStroke(BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_BEVEL, 1));
		Color fgCol = new Color(wizard.fgColorMagnifier);
		TreeNode firstNode, secondNode;

		if (image != null) {
			try {
				lensImage = image.getSubimage(imgX - 9, imgY - 9, 38, 38);
				// Necessary for images with alpha channel
				g2.setColor(new Color(wizard.transparentShade));
				g2.fillRect(0, 0, getWidth(), getHeight());

				if (wizard.showSourceImageInMagnifier) {
					lensImageSrc = imageBuffer.getBlendImage().getSubimage(
							imgX - 9, imgY - 9, 38, 38);
					g2.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER, wizard.transparencyRatio));
				}

				g2.drawImage(lensImage, lensTransformOp, 0, 0);

				if (wizard.showSourceImageInMagnifier) {
					g2.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER,
							1 - wizard.transparencyRatio));
					g2.drawImage(lensImageSrc, lensTransformOp, 0, 0);
				}

				if (wizard.highlightFGColorInMagnifier) {
					// Alter the color of the foreground (color 0) pixels
					g2.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER, 1));
					for (int i = 1; i < 39; i++) {
						for (int j = 1; j < 39; j++) {
							if (image.getRGB(imgX - 10 + i, imgY - 10 + j) == BIN1) {
								g2.setColor(fgCol);
								g2.fillRect(i * factor - factor, j * factor
										- factor, factor, factor);
							}
						}
					}
				}

				{
					// Draw miniature tree nodes and branches
					g2.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER, 1));
					g2.setStroke(new BasicStroke(2, BasicStroke.CAP_BUTT,
							BasicStroke.JOIN_BEVEL));
					Shape magnifierView = new Rectangle2D.Float(imgX - 9,
							imgY - 9, 38, 38);
					nodesInMagnifier = imagePanel
							.getNodesInSelection(magnifierView);
					branchesInMagnifier = topology.getBranches();

					{
						if (branchesInMagnifier != null) {
							for (int i = 0; i < branchesInMagnifier.size(); i++) {
								Branch branch = branchesInMagnifier
										.elementAt(i);
								if (branch != null) {
									firstNode = branch.getFirstNode();
									secondNode = branch.getSecondNode();
									int x1 = firstNode.getX() - this.imgX;
									int x2 = secondNode.getX() - this.imgX;
									int y1 = firstNode.getY() - this.imgY;
									int y2 = secondNode.getY() - this.imgY;
									if ((imagePanel.getSelectedBranch()) == branch)
										g2.setColor(branchSelectedColor);
									else
										g2.setColor(branchColor);

									if (wizard.showAllBranches)
										g2.draw(new Line2D.Float(factor * x1
												+ middleX + 4, factor * y1
												+ middleY + 4, factor * x2
												+ middleX + 4, factor * y2
												+ middleY + 4));
									else if (wizard.useOneNodeMode)// Draw the
									// real path
									{
										if ((firstNode == imagePanel
												.getSelectedNode())
												|| (secondNode == imagePanel
														.getSelectedNode())) {
											g2.setColor(Color.orange);
											Vector<Point> path = branch
													.getPath(); // Draw the
											// "real" path
											if (path != null) {
												Iterator<Point> it = path
														.iterator();
												while (it.hasNext()) {
													Point p = (Point) it.next();
													int xc = p.x - this.imgX;
													int yc = p.y - this.imgY;
													g2
															.fill(new Rectangle2D.Float(
																	factor
																			* xc
																			+ middleX,
																	factor
																			* yc
																			+ middleY,
																	8, 8));
												}
											}
										}
									} else if (wizard.showTracedPaths) {
										g2.setStroke(new BasicStroke(1,
												BasicStroke.CAP_BUTT,
												BasicStroke.JOIN_BEVEL));

										int numSegments = branch
												.getNumSegments();

										Vector<Integer> lengthRelevant = branch
												.getSegLengthRelevant();
										for (int j = 0; j < numSegments; j++) {
											int relevancy = lengthRelevant
													.elementAt(j);
											if (relevancy == RELEVANT)
												g2.setColor(new Color(210, 100,
														100));
											else if (relevancy == IRRELEVANT)
												g2.setColor(Color.black);
											else
												g2.setColor(Color.lightGray);

											Vector<Point> points = branch
													.getPathSegmentPoints(j);
											if (points != null) {
												Iterator<Point> pit = points
														.iterator();
												while (pit.hasNext()) {
													Point p = (Point) pit
															.next();
													int xc = p.x - this.imgX;
													int yc = p.y - this.imgY;
													g2
															.fill(new Rectangle2D.Float(
																	factor
																			* xc
																			+ middleX,
																	factor
																			* yc
																			+ middleY,
																	8, 8));
												}
											}
										}
									}
								}
							}
						}
					}

					if (nodesInMagnifier != null) {
						for (int i = 0; i < nodesInMagnifier.size(); i++) {
							TreeNode node = nodesInMagnifier.elementAt(i);
							if (node != null) {
								if (imagePanel.getSelectedNode() == node)
									g2.setColor(selectionColor);
								else
									g2.setColor(Color.green);
								int xc = node.getX() - this.imgX;
								int yc = node.getY() - this.imgY;
								g2
										.fill(new Rectangle2D.Float(factor * xc
												+ middleX, factor * yc
												+ middleY, 8, 8));
							}
						}
					}
				}

				// Draw Center Rectangle
				g2.setStroke(new BasicStroke(3, BasicStroke.CAP_BUTT,
						BasicStroke.JOIN_BEVEL));
				g2.setComposite(AlphaComposite.getInstance(
						AlphaComposite.SRC_OVER, 1));
				g2.setColor(Color.white);
				g2.draw(line1);
				g2.draw(line2);

				g2.setStroke(new BasicStroke(BasicStroke.CAP_BUTT,
						BasicStroke.JOIN_BEVEL, 1));
				g2.setColor(Color.red);
				g2.draw(line1);
				g2.draw(line2);
			} catch (RasterFormatException e) {
			}
		}
	}

	public void clear() {
		Graphics2D g2 = (Graphics2D) this.getGraphics();
		g2.setColor(new Color(defaultTransparencyShade));
		g2.fillRect(0, 0, getWidth(), getHeight());
	}

	public Dimension getPreferredSize() {
		return new Dimension(144, 144);
	}

	public void setObjectReferences(ImageBuffer imageBuffer,
			ImagePanel imagePanel, Wizard wizard, TreeTopology topology) {
		this.imageBuffer = imageBuffer;
		this.imagePanel = imagePanel;
		this.wizard = wizard;
		this.topology = topology;
	}

}