/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.GUI;

import java.awt.Point;
import java.awt.Shape;
import java.awt.geom.Area;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.util.Hashtable;

import TreeSnatcher.Core.ImageOperations;

public class ImageBuffer {
	BufferedImage srcImage = null;
	BufferedImage snapshot = null;
	BufferedImage binarizedImage = null;
	Shape imageArea;
	Shape blendArea;
	GUIActions guiActions;
	ImagePanel imagePanel;
	ImageOperations imageOperations;
	Wizard wizard;

	public ImageBuffer(BufferedImage image) {
		if (image == null)
			image = new BufferedImage(800, 600, BufferedImage.TYPE_INT_RGB);
		storeSourceImage(image);
	}

	public void storeSourceImage(BufferedImage image) throws OutOfMemoryError {
		if (image != null) {
			this.srcImage = new BufferedImage(image.getColorModel(), image
					.copyData(null), image.isAlphaPremultiplied(), null);
			imageArea = new Area(new Rectangle2D.Float(0, 0, srcImage
					.getWidth(), srcImage.getHeight()));
			blendArea = imageArea;
		}
	}

	public BufferedImage getSourceImage() {
		return this.srcImage;
	}

	public void storeBinarizedImage(BufferedImage image)
			throws OutOfMemoryError {
		if (image != null) {
			this.binarizedImage = new BufferedImage(image.getColorModel(),
					image.copyData(null), image.isAlphaPremultiplied(), null);
		}
	}

	public BufferedImage getBinarizedImage(BufferedImage image) {
		return this.binarizedImage;
	}

	public void storeAsSnapshot(BufferedImage image) throws OutOfMemoryError {
		if (image != null) {
			this.snapshot = new BufferedImage(image.getColorModel(), image
					.copyData(null), image.isAlphaPremultiplied(), null);
		}
	}

	public BufferedImage copyFromSnapshot() throws OutOfMemoryError {
		if (snapshot != null) {
			BufferedImage image = new BufferedImage(snapshot.getColorModel(),
					snapshot.copyData(null), snapshot.isAlphaPremultiplied(),
					null);
			return image;
		} else {
			System.out.println("Snapshot is empty!");
			return new BufferedImage(srcImage.getColorModel(), srcImage
					.copyData(null), srcImage.isAlphaPremultiplied(), null);
		}
	}

	public BufferedImage cloneSourceImage() throws OutOfMemoryError {
		if (srcImage != null) {
			return new BufferedImage(srcImage.getColorModel(), srcImage
					.copyData(null), srcImage.isAlphaPremultiplied(), null);
		}
		return null;
	}

	public BufferedImage getProcessedImage() {
		return imagePanel.getCurrentImage();
	}

	public BufferedImage cloneProcessedImage() throws OutOfMemoryError {
		BufferedImage processedImage = getProcessedImage();
		if (processedImage != null) {
			return new BufferedImage(processedImage.getColorModel(),
					processedImage.copyData(null), processedImage
							.isAlphaPremultiplied(), null);
		}
		return null;
	}

	

	public BufferedImage getBlendImage() {
		if (srcImage != null)
			return srcImage;
		else
			return null;
	}

	public BufferedImage getBorderlessImage() {
		BufferedImage image = imageOperations.removeBorderFromImage(imagePanel
				.getCurrentImage());
		return image;
	}

	public BufferedImage getBorderlessImage(BufferedImage img) {
		BufferedImage image = imageOperations.removeBorderFromImage(img);
		return image;
	}

	public BufferedImage getBinarizedImage() {
		if (binarizedImage != null)
			return this.binarizedImage;
		else
			return null;
	}

	public void setObjectReferences(GUIActions ga, ImagePanel ip,
			ImageOperations io) {
		this.guiActions = ga;
		this.imagePanel = ip;
		this.imageOperations = io;
	}

	

	public void cropBlendImage(Shape selection) {
		// As Undo is not possible after a crop operation, there no need to
		// store
		// the original image any longer. Therefore it can also be cropped
		srcImage = imageOperations.getSubImage(srcImage, selection, true);
	}

	public int[][] returnImageAs2DArray(BufferedImage image)
			throws OutOfMemoryError {
		int w = image.getWidth();
		int h = image.getHeight();
		int[][] rgb = new int[w][h];

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				rgb[x][y] = image.getRGB(x, y);
			}
		}

		return rgb;
	}

	public BufferedImage return2DArrayAsImage(int[][] rgb)
			throws OutOfMemoryError {
		int h = rgb[0].length;
		int w = rgb.length;
		BufferedImage image = new BufferedImage(w, h,
				BufferedImage.TYPE_INT_ARGB);

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				image.setRGB(x, y, rgb[x][y]);
			}
		}

		return image;
	}

	public void makeRGBFromIndexTable(int[][] rgb, int[] indexedPalette) {
		int h = rgb[0].length;
		int w = rgb.length;

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				rgb[x][y] = indexedPalette[rgb[x][y]];
			}
		}
	}

	

	public BufferedImage returnRGBArrayAsImage(int[] rgb, int w, int h)
			throws OutOfMemoryError {
		BufferedImage image = new BufferedImage(w, h,
				BufferedImage.TYPE_INT_ARGB);

		for (int x = w; x-- > 0;) {
			for (int y = h; y-- > 0;) {
				image.setRGB(x, y, rgb[y * w + x]);
			}
		}
		return image;
	}

	// Returns a clone of an area of BufferedImage img
	private BufferedImage getClonedImageArea(BufferedImage img, Point m, int r) {
		BufferedImage temp = img.getSubimage(m.x - r, m.y - r, 2 * r + 1,
				2 * r + 1);

		String[] pnames = temp.getPropertyNames();
		Hashtable<String, Object> cproperties = new Hashtable<String, Object>();
		if (pnames != null) {
			for (int j = 0; j < pnames.length; j++) {
				cproperties.put(pnames[j], temp.getProperty(pnames[j]));
			}
		}
		WritableRaster wr = temp.getRaster();
		WritableRaster cwr = wr.createCompatibleWritableRaster();
		cwr.setRect(wr);

		return new BufferedImage(temp.getColorModel(), cwr, temp
				.isAlphaPremultiplied(), cproperties);
	}

	public void reset() {
		blendArea = imageArea;
	}
}
