/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import TreeSnatcher.GUI.GUIActions;
import TreeSnatcher.GUI.ImageBuffer;
import TreeSnatcher.GUI.ImagePanel;
import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.Utils.NumberUtility;

public class TreeTopology implements Constants {
	private static final long serialVersionUID = 1L;
	private ImageOperations imageOperations;
	private ImagePanel imagePanel;
	private ImageBuffer imageBuffer;
	private Wizard wizard;
	private NewickCalculator newickCalculator;
	private MainWindow mainWindow;

	Graphics2D g2;
	BufferedImage binarizedImage;
	Vector<TreeNode> nodes = new Vector<TreeNode>();
	Vector<TreeNode> innerNodes = new Vector<TreeNode>();
	Vector<TreeNode> tips = new Vector<TreeNode>();
	Vector<Branch> branches = new Vector<Branch>();
	HashMap<Point, HashMap<Point, Integer>> distances;
	Vector<Branch> tempBranches = new Vector<Branch>();

	static ObjectOutputStream os = null;
	static ByteArrayOutputStream out = null;

	public TreeTopology(ImageOperations iop, ImagePanel ip, ImageBuffer ib,
			Wizard wiz, MainWindow mainWindow, GUIActions guiActions) {
		this.imageOperations = iop;
		this.imagePanel = ip;
		this.imageBuffer = ib;
		this.wizard = wiz;
		this.mainWindow = mainWindow;
		newickCalculator = new NewickCalculator(nodes, branches, ip, wizard,
				this, imageOperations);
		binarizedImage = imageBuffer.getBinarizedImage();
	}

	public void collectTips(int fgColor) {
		Vector<Point> points = imageOperations.getNNeighborPixels(imagePanel
				.getCurrentImage(), 1, wizard.wholeImage, fgColor);
		for (int i = 0; i < points.size(); i++) {
			Point p = (Point) points.elementAt(i);
			Tip t = new Tip(p.x, p.y, false);

			tips.add(t);
		}
		imagePanel.repaint();
		cleanupNodes(tips);
		points = null;
	}

	public void collectNodes(int fgColor) {
		// Collect candidate locations for inner nodes and tips
		Vector<Point> points = null, temp = new Vector<Point>();
		Point relPoint, nextPoint;
		points = imageOperations.getNNeighborPixels(imagePanel
				.getCurrentImage(), 3, wizard.wholeImage, fgColor);
		points.addAll(imageOperations.getNNeighborPixels(imagePanel
				.getCurrentImage(), 4, wizard.wholeImage, fgColor));

		do {

			relPoint = points.firstElement();
			temp.add(relPoint);

			// Find all direct neighbors of relPoint, store them in temp
			for (int j = 0; j < points.size(); j++) {
				nextPoint = points.elementAt(j);
				if ((!relPoint.equals(nextPoint) && (relPoint
						.distance(nextPoint) < 2.5))) {
					temp.add(nextPoint);
				}
			}

			int size = temp.size();

			if ((size == 1)) // A single candidate point or two candidates
			{
				Point p = (Point) temp.firstElement();
				InnerNode in = new InnerNode(p.x, p.y, false);
				innerNodes.add(in);
			} else // Three or more neighboring candidate points
			{
				double sumX = 0, sumY = 0, meanX, meanY;

				for (int k = 0; k < size; k++) {
					sumX = sumX + temp.elementAt(k).x;
					sumY = sumY + temp.elementAt(k).y;
				}
				meanX = Math.ceil(sumX / size);
				meanY = Math.ceil(sumY / size);

				// Select the point that is nearest to location (meanX, meanY)
				int shade = imagePanel.getCurrentImage().getRGB((int) meanX,
						(int) meanY);
				if ((shade == BIN1) || (shade == wizard.treeFloodColor)) // There
				// is
				// an
				// ideal
				// foreground
				// pixel
				{
					InnerNode in = new InnerNode((int) meanX, (int) meanY,
							false);
					innerNodes.add(in);
				} else // The ideal pixel is a background pixel
				{
					// Instead, accept the first neighbor as inner node
					Point p = temp.firstElement();
					InnerNode in = new InnerNode(p.x, p.y, false);
					innerNodes.add(in);
				}
			}
			points.removeAll(temp);
			points.trimToSize();
			temp.clear();
		} while (!points.isEmpty());

		points = null;
		cleanupNodes(innerNodes);
		imagePanel.repaint();
	}

	public void wipeOutDoublettes() {
		// Remove those newly set nodes that are too near to nodes the the user
		// created previously
		Vector<TreeNode> userCreatedNodes = new Vector<TreeNode>();
		// Collect user created nodes
		for (int i = 0; i < nodes.size(); i++) {
			TreeNode node = (TreeNode) nodes.elementAt(i);
			if (node.getUserGenerated())
				userCreatedNodes.add(node);
		}

		// Compare user created nodes and all other nodes; delete those which
		// are too near to user created nodes
		Iterator<TreeNode> nodesIterator = nodes.iterator();
		while (nodesIterator.hasNext()) {
			TreeNode node = (TreeNode) nodesIterator.next();
			if (node.getUserGenerated() == false) // Node is program created
			{
				Iterator<TreeNode> ucnIterator = userCreatedNodes.iterator();
				while (ucnIterator.hasNext()) {
					TreeNode userCreatedNode = (TreeNode) ucnIterator.next();

					if (userCreatedNode.dist2(node) <= 2) // Nodes are too near
					{
						if (node instanceof InnerNode) {
							innerNodes.remove(node);
							nodes.remove(node);
						} else if (node instanceof Tip) {
							tips.remove(node);
							nodes.remove(node);
						}
					}
				}
			}
		}
		userCreatedNodes = null;
	}

	public boolean collectBranches() {
		boolean couldBuildTopology = false;
		;
		tempBranches.clear();

		// Ensure that there are InnerNode- and Tip-objects
		Iterator<TreeNode> tit = nodes.iterator();
		int is = 0, os = 0;
		while (tit.hasNext()) {
			TreeNode node = tit.next();
			if (node instanceof InnerNode)
				is++;
			else
				os++;
		}

		if ((is >= 1) && (os >= 2)) // For a sensible topology, there must be at
		// least one inner node and two tips
		{
			// Color the node locations as they need to be distinguished from
			// other foreground pixels
			imageOperations.colorNodePositions(nodes, Color.red);

			try {
				// Build the distance data structures for each node
				// {Distances} contains the distances between path pixels and
				// the node position from which the path emerges
				distances = new HashMap<Point, HashMap<Point, Integer>>();
				if (wizard.useElaboratePathFinding)
					imageOperations.buildDistancesElaborate(nodes, distances);
				else
					imageOperations.buildDistancesSimple(nodes, distances);

				// Follow the paths in a greedy fashion in reverse to find the
				// paths with the shortest distances between the nodes
				Iterator<TreeNode> nodesIt = nodes.iterator();

				while (nodesIt.hasNext()) {
					TreeNode seedNode = nodesIt.next();
					couldBuildTopology = imageOperations.buildPath(seedNode,
							nodes, distances, tempBranches);
				}

				if (couldBuildTopology) {
					for (int i = 0; i < tempBranches.size(); i++) {
						Branch b = tempBranches.elementAt(i);
						if (((b.getFirstNode() != null) && (b.getSecondNode() != null))
								&& (!b.getFirstNode().equals(b.getSecondNode()))) {
							if ((b.getFirstNode() instanceof Tip)
									|| (b.getSecondNode() instanceof Tip))
								b.setOuterBranch(true);
							else
								b.setOuterBranch(false);
							addBranch(b);
						}
					}

					tempBranches.clear();

					if ((branches != null) && (branches.size() > 0)) {
						// Get and use the inflection points on the branches
						imageOperations.identifyInflectionPoints(branches,
								nodes);
						imageOperations
								.determineBranchSegments(branches, nodes);
						imageOperations.calculateSegmentSlope(branches);
						imageOperations.determineLengthRelevantSegments(
								branches, nodes);
					}
				}
			} catch (Exception ex) {
				imagePanel.repaint();
			}

		}

		wizard.topologyDeterminationFinished = couldBuildTopology;
		imageOperations.colorNodePositions(nodes, Color.black);
		return couldBuildTopology;
	}

	public boolean foregroundPathExists(TreeNode n1, TreeNode n2) {
		// Get distance information for both nodes
		HashMap<Point, Integer> dist_n1 = distances.get(n1.getLocation());
		HashMap<Point, Integer> dist_n2 = distances.get(n2.getLocation());
		if ((dist_n1 != null) && (dist_n2 != null)) {
			if ((dist_n1.containsKey(n2.getLocation()))
					&& (dist_n2.containsKey(n1.getLocation())))
				return true;
		}
		return false;
	}

	public int getPathLengthInPixels(TreeNode n1, TreeNode n2) {
		int length = -1;
		// Get distance data structures for both nodes
		HashMap<Point, Integer> dist_n1 = distances.get(n1.getLocation());
		HashMap<Point, Integer> dist_n2 = distances.get(n2.getLocation());
		if ((dist_n1 != null) && (dist_n2 != null)) {
			length = (Integer) dist_n1.get(n2.getLocation());
		}
		return length;
	}

	public TreeNode getNodeAt(Point p) {
		TreeNode node = null;
		for (int i = 0; i < nodes.size(); i++) {
			node = (TreeNode) nodes.elementAt(i);
			if (node.getLocation().equals(p))
				return node;
		}
		return null;
	}

	public Vector<Branch> getBranchesAtNode(TreeNode node) {
		Vector<Branch> selectedBranches = new Vector<Branch>();
		for (int i = 0; i < branches.size(); i++) {
			Branch branch = (Branch) branches.elementAt(i);
			if ((branch.getFirstNode() == node)
					|| (branch.getSecondNode() == node))
				selectedBranches.addElement(branch);
		}
		return selectedBranches;
	}

	public Vector<Branch> getInnerBranches() {
		Vector<Branch> innerBranches = new Vector<Branch>();
		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch b = it.next();
			if ((b.getFirstNode() instanceof InnerNode)
					&& (b.getSecondNode() instanceof InnerNode))
				innerBranches.add(b);
		}

		return innerBranches;
	}

	public Vector<Branch> getOuterBranches() {
		Vector<Branch> outerBranches = new Vector<Branch>();
		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch b = it.next();
			if ((b.getFirstNode() instanceof Tip)
					|| (b.getSecondNode() instanceof Tip))
				outerBranches.add(b);
		}

		return outerBranches;
	}

	public Branch getBranchBetweenNodes(TreeNode firstNode, TreeNode secondNode) {
		for (int i = 0; i < branches.size(); i++) {
			Branch branch = (Branch) branches.elementAt(i);
			if (((branch.getFirstNode() == firstNode) && (branch
					.getSecondNode() == secondNode))
					|| ((branch.getFirstNode() == secondNode) && (branch
							.getSecondNode() == firstNode)))
				return branch;
		}
		return null;
	}

	public Vector<InnerNode> getInnerNodes() {
		Vector<InnerNode> innerNodes = new Vector<InnerNode>();
		Iterator<TreeNode> it = nodes.iterator();
		while (it.hasNext()) {
			TreeNode treeNode = it.next();
			if (treeNode instanceof InnerNode)
				innerNodes.add((InnerNode) treeNode);
		}
		return innerNodes;
	}

	public Branch getBranch(int id) {
		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch b = it.next();
			if (b.getId() == id)
				return b;
		}
		return null;
	}

	public boolean branchExists(Branch branch) {

		if (branches.contains(branch))
			return true;
		else
			return false;
	}

	public void moveNode(TreeNode node, Point p) {
		Iterator<TreeNode> nodesIt = nodes.iterator();
		while (nodesIt.hasNext()) {
			TreeNode n = (TreeNode) nodesIt.next();
			if (n.equals(node))
				n.setLocation(p);
		}
	}

	public void removeAllNodes() {
		if (nodes != null) {
			Iterator<TreeNode> nodesIt = nodes.iterator();
			while (nodesIt.hasNext()) {
				TreeNode node = (TreeNode) nodesIt.next();
				if (node.getUserGenerated()) {
					if (wizard.mayDiscardUserGeneratedObjects) {
						removeConnectedBranches(node);
						nodesIt.remove();
					}
				} else {
					removeConnectedBranches(node);
					nodesIt.remove();
				}
			}
		}
		wizard.nodesCollected = false;
		wizard.referenceNodeId = -1;
		wizard.topologyDeterminationFinished = false;

		TreeNode.signalNode = -1;
		innerNodes.removeAllElements();
		tips.removeAllElements();
	}

	public void removeAllBranches() {
		if (branches != null) {
			Iterator<Branch> branchesIt = branches.iterator();
			while (branchesIt.hasNext()) {
				Branch branch = (Branch) branchesIt.next();
				if (branch.getUserGenerated()) {
					// Remove the branch object from the Vector
					// Use removeBranch() as the nodes' neighborship information
					// must be updated
					if (wizard.mayDiscardUserGeneratedObjects) {
						removeBranch(branch);
					}
					;
				} else {
					branchesIt.remove();
				}
				// Do not use removeBranch() as the nodes' neighborship
				// information shall be preserved
			}
			wizard.referenceNodeId = -1;
			wizard.topologyDeterminationFinished = false;
			TreeNode.signalNode = -1;
		}
	}

	// Removes a node and its accompanying branches
	// The neighboring nodes' neighborship information gets adjusted
	// (removeConnectedBranches)
	// Recalculate the tree topology
	public void removeNode(TreeNode node) {
		Iterator<TreeNode> nodesIt = nodes.iterator();
		while (nodesIt.hasNext()) {
			if (node.equals((TreeNode) nodesIt.next())) {
				removeConnectedBranches(node);
				if (node.id == wizard.referenceNodeId)
					wizard.referenceNodeId = -1;
				nodesIt.remove();
			}
		}
		if (nodes.isEmpty()) {
			wizard.topologyDeterminationFinished = false;
			TreeNode.signalNode = -1;
		}

		redoTopology();
	}

	// Removes the nodes within current rectangular selection and their
	// accompanying branches
	// The neighboring nodes' neighborship information gets adjusted
	// (removeConnectedBranches)
	// Recalculate the tree topology
	public void removeNodes(Vector<TreeNode> nodesInSelection) {
		for (int i = 0; i < nodesInSelection.size(); i++) {
			TreeNode nodeInSelection = (TreeNode) nodesInSelection.elementAt(i);
			Iterator<TreeNode> nodesIt = nodes.iterator();
			while (nodesIt.hasNext()) {
				if (nodeInSelection.equals((TreeNode) nodesIt.next())) {
					removeConnectedBranches(nodeInSelection);
					nodesIt.remove();
				}
			}
		}
		if (nodes.isEmpty()) {
			wizard.topologyDeterminationFinished = false;
			TreeNode.signalNode = -1;
		}

		redoTopology();
	}

	// Removes a branch
	// Purges for its two defining nodes the mutual reference in their
	// neighborship objects
	// Recalculates the tree topology
	public void removeBranch(Branch branch) {
		Iterator<Branch> branchesIt = branches.iterator();
		while (branchesIt.hasNext()) {
			if (branch.equals((Branch) branchesIt.next())) {
				TreeNode firstNode = branch.getFirstNode();
				Vector<TreeNode> firstNeighbors = firstNode.getNeighbors();
				TreeNode secondNode = branch.getSecondNode();
				Vector<TreeNode> secondNeighbors = secondNode.getNeighbors();

				if (firstNode.id == wizard.referenceNodeId)
					wizard.referenceNodeId = -1;
				if (secondNode.id == wizard.referenceNodeId)
					wizard.referenceNodeId = -1;
				firstNeighbors.remove(secondNode);
				secondNeighbors.remove(firstNode);

				// Remove the very branch
				branchesIt.remove();
			}
		}

		if (branches.isEmpty()) {
			wizard.topologyDeterminationFinished = false;
			TreeNode.signalNode = -1;
		}

		redoTopology();
		NumberUtility.setNumFractionDigits(branches);
	}

	// Removes the branches within current rectangular selection and their
	// accompanying nodes
	// Purges for their two defining nodes the mutual reference in their
	// neighborship objects
	// Recalculates the tree topology
	public void removeBranches(Vector<Branch> branchesInSelection) {
		for (int j = 0; j < branchesInSelection.size(); j++) {
			Branch branchInSelection = (Branch) branchesInSelection
					.elementAt(j);
			Iterator<Branch> branchesIt = branches.iterator();
			while (branchesIt.hasNext()) {
				if (branchInSelection.equals((Branch) branchesIt.next())) {
					TreeNode firstNode = branchInSelection.getFirstNode();
					TreeNode secondNode = branchInSelection.getSecondNode();

					if (firstNode.id == wizard.referenceNodeId)
						wizard.referenceNodeId = -1;
					if (secondNode.id == wizard.referenceNodeId)
						wizard.referenceNodeId = -1;
					firstNode.getNeighbors().remove(secondNode);
					secondNode.getNeighbors().remove(firstNode);

					// Remove the very branch
					branchesIt.remove();
					redoTopology();
				}
			}
		}

		if (branches.isEmpty()) {
			wizard.topologyDeterminationFinished = false;
			TreeNode.signalNode = -1;
		}
		;
		NumberUtility.setNumFractionDigits(branches);
	}

	// Remove the branches which are connected to this node
	private void removeConnectedBranches(TreeNode node) {
		TreeNode firstNode, secondNode;

		Iterator<Branch> branchesIt = branches.iterator();
		while (branchesIt.hasNext()) {
			Branch branch = (Branch) branchesIt.next();
			firstNode = branch.getFirstNode();
			secondNode = branch.getSecondNode();
			if ((firstNode.equals(node)) || (secondNode.equals(node))) {
				if (firstNode.id == wizard.referenceNodeId)
					wizard.referenceNodeId = -1;
				if (secondNode.id == wizard.referenceNodeId)
					wizard.referenceNodeId = -1;
				firstNode.getNeighbors().remove(secondNode);
				secondNode.getNeighbors().remove(firstNode);

				branchesIt.remove();
			}
		}

		if (branches.isEmpty()) {
			wizard.topologyDeterminationFinished = false;
			TreeNode.signalNode = -1;
		}
		;
		NumberUtility.setNumFractionDigits(branches);
	}

	private void cleanupNodes(Vector<TreeNode> treeNodes) {
		// TODO: Determine dmin based on the local foreground in the BINARIZED
		// (not thinned) image
		double dmin = 3;
		if (wizard.useElaboratePathFinding)
			dmin = 3;
		double dmin2 = dmin * dmin;
		Object[] nodeArray = treeNodes.toArray();
		Vector<TreeNode> goodNodes = new Vector<TreeNode>(treeNodes.size());
		for (int i = 0; i < nodeArray.length; i++) {
			if (nodeArray[i] != null) {
				TreeNode n1 = (TreeNode) nodeArray[i];
				goodNodes.add(n1);

				// Delete all remaining nodes close to n1
				for (int j = i + 1; j < nodeArray.length; j++) {
					if (nodeArray[j] != null) {
						TreeNode n2 = (TreeNode) nodeArray[j];
						if (n1.dist2(n2) < dmin2)
							nodeArray[j] = null;
					}
				}
			}
		}
		treeNodes = null;
		nodeArray = null;
		nodes.addAll(goodNodes);
		goodNodes = null;
	}

	public InnerNode addInnerNode(Point l) {
		InnerNode in = null;
		boolean insert = true;
		if (l != null) {
			in = new InnerNode(l.x, l.y, true);

			if (nodes.size() > 0) {
				Iterator<TreeNode> nodesIt = nodes.iterator();

				while (nodesIt.hasNext()) {
					TreeNode n = nodesIt.next();
					int x = n.getX();
					int y = n.getY();
					if ((Math.abs(x - l.x) < 2) && (Math.abs(y - l.y) < 2))
						insert = false;
				}
			}
			if (insert)
				nodes.add(in);
		}

		if (wizard.topologyDeterminationFinished)
			redoTopology();
		return in;
	}

	public Tip addTip(Point l) {
		Tip t = null;
		boolean insert = true;
		if (l != null) {
			t = new Tip(l.x, l.y, true);

			if (nodes.size() > 0) {
				Iterator<TreeNode> nodesIt = nodes.iterator();
				while (nodesIt.hasNext()) {
					TreeNode n = nodesIt.next();
					int x = n.getX();
					int y = n.getY();
					if ((Math.abs(x - l.x) < 2) && (Math.abs(y - l.y) < 2))
						insert = false;
				}
			}
			if (insert)
				nodes.add(t);
		}

		if (wizard.topologyDeterminationFinished)
			redoTopology();
		return t;
	}

	public void addBranch(Branch branch) {
		boolean duplicateBranch = false;
		boolean tipWithMultipleBranches = false;
		if (branch != null) {
			// Do not add a branch that already exists
			TreeNode firstNode = branch.getFirstNode();
			TreeNode secondNode = branch.getSecondNode();
			for (int i = 0; i < branches.size(); i++) {
				Branch branch2 = (Branch) branches.elementAt(i);
				TreeNode firstNode2 = branch2.getFirstNode();
				TreeNode secondNode2 = branch2.getSecondNode();

				// Avoid duplicate branches
				if (((firstNode.equals(firstNode2)) && (secondNode
						.equals(secondNode2)))
						|| ((firstNode.equals(secondNode2)) && (secondNode
								.equals(firstNode2))))
					duplicateBranch = true;

				// Avoid Tip with more than one emerging branch
				if ((firstNode2 instanceof Tip) && (firstNode instanceof Tip)
						&& (firstNode.equals(firstNode2)))
					tipWithMultipleBranches = true;
				else if ((firstNode2 instanceof Tip)
						&& (secondNode instanceof Tip)
						&& (secondNode.equals(firstNode2)))
					tipWithMultipleBranches = true;
				else if ((secondNode2 instanceof Tip)
						&& (firstNode instanceof Tip)
						&& (firstNode.equals(secondNode2)))
					tipWithMultipleBranches = true;
				else if ((secondNode2 instanceof Tip)
						&& (secondNode instanceof Tip)
						&& (secondNode.equals(secondNode2)))
					tipWithMultipleBranches = true;
			}
			if ((!duplicateBranch) && (!tipWithMultipleBranches)) {
				branches.add(branch);
				firstNode.addNeighbor(secondNode);
				secondNode.addNeighbor(firstNode);
			}

			if (!wizard.branchesCollecting)
				redoTopology(); // Only recalculate the topology after a manual
			// branch addition
		}
	}

	public InnerNode insertRootAtBranch(Branch branch) {
		if (branch == null)
			return null;
		// Get the path of this branch
		Vector<Point> path = branch.getPath();

		// Go to the middle of the branch
		Point p = null;
		int halfway = (int) (0.5 * (branch.getTotalLengthInPixels() + 1));
		for (int i = 0; i < halfway; i++)
			p = path.elementAt(i);
		InnerNode in = new InnerNode(p, true);
		nodes.add(in);
		wizard.referenceNodeId = in.id;
		redoTopology();

		return in;
	}

	public void reset() {
		TreeNode.num = 0;
		Branch.num = 0;
		wizard.referenceNodeId = -1;

		nodes.removeAllElements();
		branches.removeAllElements();
		innerNodes.removeAllElements();
		tips.removeAllElements();
	}

	public Vector<TreeNode> getNodes() {
		if (nodes == null)
			return null;
		else
			return this.nodes;
	}

	public void setNodes(Vector<TreeNode> nodes) {
		this.nodes = nodes;
	}

	public Vector<Branch> getBranches() {
		if (branches == null)
			return null;
		else
			return this.branches;
	}

	public void setBranches(Vector<Branch> branches) {
		this.branches = branches;
	}

	// Returns the lowest pairwise distance over all TreeNode objects
	public int getLowestPairwiseDistance() {
		// Naive approach, better use a Sweep-line-approach
		double minDistance = Integer.MAX_VALUE;

		for (int i = 0; i < this.getNodes().size(); i++) {
			for (int j = 0; j < nodes.size(); j++) {
				Point p1 = new Point(nodes.elementAt(i).getLocation());
				Point p2 = new Point(nodes.elementAt(j).getLocation());

				if (!p1.equals(p2)) {
					double dist = p1.distance(p2);
					if (dist < minDistance)
						minDistance = dist;
				}
			}
		}

		if (minDistance == Integer.MAX_VALUE)
			return 0;
		else
			return (int) minDistance;
	}

	public String transformIntoNewickExpression() {
		// Mark all branches as "not visited"
		for (int i = 0; i < branches.size(); i++) {
			((Branch) branches.elementAt(i)).setVisited(false);
		}
		// Mark all nodes as "not visited"
		for (int i = 0; i < nodes.size(); i++) {
			((TreeNode) nodes.elementAt(i)).setVisited(false);
		}

		if (nodes.size() == 0)
			return " The tree nodes need to be determined.";
		else if (branches.size() == 0)
			return "The branches need to be determined.";
		else {
			if ((wizard.topologyDeterminationFinished == true)
					&& (wizard.mostRecentTopologyError != 0)) {
				wizard.showResultFrame = true;
				mainWindow.swapWizardAndNewickPanels();
			}

			// Determine the Newick expression for the tree topology
			return newickCalculator.calculateNewickExpression(nodes, branches);
		}
	}

	// Redo the topology check, display it on-screen
	private void redoTopology() {
		transformIntoNewickExpression();
		imagePanel.updateResultFrame();
	}

	public void redisplayNewickExpression() {
		try {
			newickCalculator.calculateNewickExpression(nodes, branches);
			imagePanel.updateResultFrame();
		} catch (NullPointerException e) {
			System.out.println("Newick Expression: Exception");
		}
		;
	}

	public String getNewickExpression() {
		return newickCalculator.getNewickString();
	}

	public void setTaxonName(int id, String name) {
		Iterator<TreeNode> nodesIt = nodes.iterator();

		while (nodesIt.hasNext()) {
			TreeNode n = nodesIt.next();
			if (id == n.getId()) {
				((Tip) n).setTaxonName(name);
				imagePanel.repaint();
			}
		}
	}

	public boolean findNode(int id) {
		Iterator<TreeNode> nodesIt = nodes.iterator();

		while (nodesIt.hasNext()) {
			TreeNode n = nodesIt.next();
			if (id == n.getId())
				return true;
		}
		return false;
	}

	public void issueSnapshot(Vector<Branch> branches, Vector<TreeNode> nodes) {
		this.branches = branches;
		this.nodes = nodes;
		redoTopology();
	}

	public ByteArrayOutputStream getSerializedTopology() {
		// Store the current topology in memory
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		ObjectOutputStream os = null;
		try {
			os = new ObjectOutputStream(out);
			os.writeObject(nodes);
			os.writeObject(branches);
			os.writeObject(TreeNode.num);
			os.writeObject(TreeNode.signalNode);
			os.writeObject(Branch.num);
			os.writeObject(Tip.nameCnt);
			os.flush();
		} catch (IOException e) {
		}
		return out;
	}

}
