/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.Point;
import java.awt.Shape;
import java.awt.geom.Ellipse2D;
import java.io.Serializable;
import java.util.Iterator;
import java.util.Vector;

import TreeSnatcher.GUI.Wizard;

public class TreeNode implements Serializable, Comparable<Object>, Constants {
	private static final long serialVersionUID = 1L;
	public static int num = 1, z = 1; // num is increased during every
	// instantiation of TreeNode
	public static int signalNode = -1;
	public int flag = UNTOUCHED;
	private static int firstId_Highlighted = -1;
	private static int secondId_Highlighted = -1;
	protected int id;
	private int x;
	private int y;
	private int n; // Number of foreground neighbors
	private boolean userGenerated = false;
	private transient boolean visited = false;
	private Vector<TreeNode> neighbors = new Vector<TreeNode>();
	private static int[] offset = { 1, 2, 3, 4, 5, 4, 3, 2 };
	private int oi = 0;

	public TreeNode(Point p) {
		this.id = num++;
		this.x = p.x;
		this.y = p.y;
	}

	public TreeNode(int x, int y) {
		this.id = num++;
		this.x = x;
		this.y = y;
	}

	public TreeNode(Point p, boolean b) {
		this.id = num++;
		this.x = p.x;
		this.y = p.y;
		userGenerated = b;
	}

	public TreeNode(int x, int y, boolean b) {
		this.id = num++;
		this.x = x;
		this.y = y;
		userGenerated = b;
	}

	public void addNeighbor(TreeNode node) {
		if (neighbors.contains(node) == false)
			neighbors.add(node);
	}

	public Point getLocation() {
		return new Point(this.x, this.y);
	};

	public void setLocation(Point p) {
		this.x = p.x;
		this.y = p.y;
	};

	public int getX() {
		return this.x;
	};

	public int getY() {
		return this.y;
	};

	public int getId() {
		return this.id;
	};

	public boolean getUserGenerated() {
		return this.userGenerated;
	};

	public void setVisited(boolean b) {
		this.visited = b;
	};

	public boolean getVisited() {
		return this.visited;
	};

	public Vector<TreeNode> getNeighbors() {
		return this.neighbors;
	};

	public void resetNeighbors() {
		this.neighbors.clear();
	};

	public Shape getShape() {
		return new Ellipse2D.Double(x - 3, y - 3, 7, 7);
	}

	public Shape getSmallShape() {
		int of = (int) Math.floor(0.5 * Wizard.smallNodeRadius);
		return new Ellipse2D.Double(x - of, y - of, 2 * of + 1, 2 * of + 1);
	}

	public Shape getRootShape() {
		int of = (int) Math.floor(0.5 * Wizard.referenceNodeRadius);
		return new Ellipse2D.Double(x - of, y - of, 2 * of + 1, 2 * of + 1);
	}

	public Shape getSignalShape() {
		if (oi == 8)
			oi = 0;
		int z = offset[oi++];
		return new Ellipse2D.Double(x - z, y - z, 2 * z + 1, 2 * z + 1);
	}

	public void setUserGenerated() {
		this.userGenerated = true;
	};

	public int compareTo(Object obj) {
		TreeNode node2 = (TreeNode) obj;
		if (this.n > node2.n)
			return -1;
		if (this.n < node2.n)
			return 1;
		else
			return 0;
	}

	public double dist2(TreeNode n2) {
		int dx = this.x - n2.x;
		int dy = this.y - n2.y;
		return (dx * dx) + (dy * dy);
	}

	public boolean equals(Object o) {
		if (!(o instanceof TreeNode))
			return false;

		TreeNode t = (TreeNode) o;
		int tx = t.getX();
		int ty = t.getY();

		if ((tx == x) && (ty == y))
			return true;
		return false;
	}

	public String toString() {
		String s = "";
		if (this instanceof Tip)
			s = s + "Tip";
		else if (this instanceof InnerNode)
			s = s + "InnerNode";
		s = s + " at (" + this.x + ", " + this.y + ")";
		s = s + " with neighbors: ";
		for (int i = 0; i < neighbors.size(); i++) {
			TreeNode neighbor = neighbors.elementAt(i);
			if (neighbor instanceof Tip)
				s = s + "\n	Tip";
			else if (neighbor instanceof InnerNode)
				s = s + "\n	InnerNode";
			s = s + " at (" + neighbor.getX() + ", " + neighbor.getY() + ")";
		}
		return s;
	}

	public static void setFirstId_Highlighted(int id) {
		firstId_Highlighted = id;
	};

	public static void setSecondId_Highlighted(int id) {
		secondId_Highlighted = id;
	};

	public static int getFirstId_Highlighted() {
		return firstId_Highlighted;
	};

	public static int getSecondId_Highlighted() {
		return secondId_Highlighted;
	};
}
