/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.Color;
import java.text.NumberFormat;
import java.util.Iterator;
import java.util.Stack;
import java.util.Vector;

import TreeSnatcher.GUI.ImagePanel;
import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.Utils.NumberUtility;
import TreeSnatcher.Core.ImageOperations;

public class NewickCalculator implements Constants {
	private Vector<Branch> branches;
	private Vector<TreeNode> nodes;
	private String newickExpression;
	private Wizard wizard;
	private TreeTopology topology;
	private ImageOperations imageOperations;
	public static Vector<InnerNode> searchPath = new Vector<InnerNode>();
	public static boolean cycleFound = false;
	private NumberFormat ndf;

	public NewickCalculator(Vector<TreeNode> nodes, Vector<Branch> branches,
			ImagePanel imagePanel, Wizard wizard, TreeTopology topology,
			ImageOperations imageOperations) {
		this.branches = branches;
		this.nodes = nodes;
		this.wizard = wizard;
		this.topology = topology;
		this.imageOperations = imageOperations;
	}

	// Try to build a multifurcating tree from the information enclosed in the
	// Branch and TreeNode objects
	// If the topology is incorrect, return an error message
	// Otherwise build and return a Newick String derived from the tree
	// structure
	public String calculateNewickExpression(Vector<TreeNode> nodes,
			Vector<Branch> branches) {
		this.branches = branches;
		this.nodes = nodes;
		this.newickExpression = new String();

		// Build up a graph structure from the branches and nodes
		for (int i = 0; i < nodes.size(); i++)
			nodes.elementAt(i).setVisited(false);
		Object[] error = buildTreelikeLinkedList();

		// Process topologic errors reported from above procedure
		Integer topologyError = (Integer) error[0];
		Object errorObject = error[1];

		// Make the object blink that caused the error
		if (errorObject != null) {
			if (errorObject instanceof TreeNode) {
				TreeNode node = (TreeNode) errorObject;
				TreeNode.signalNode = node.getId();
				Branch.signalBranch = -1;
			} else if (errorObject instanceof Branch) {
				Branch branch = (Branch) errorObject;
				Branch.signalBranch = branch.getId();
				TreeNode.signalNode = -1;
			}
		}

		// Report the topology error to the Wizard
		wizard.mostRecentTopologyError = topologyError;
		// Reset some object flags as the topology must be recalculated
		if (topologyError != 0) {
			imageOperations.colorNodePositions(nodes, new Color(BIN1)); // Un-color
			// node
			// positions
			wizard.topologyDeterminationFinished = false;

			Iterator<Branch> bt = branches.iterator();
			while (bt.hasNext()) {
				Branch branch = bt.next();
				branch.visited = false;
			}

			Iterator<TreeNode> nt = nodes.iterator();
			while (nt.hasNext()) {
				TreeNode node = nt.next();
				node.setVisited(false);
				node.resetNeighbors();
			}

		}

		if (topologyError == 1)
			return "The pink tip is isolated";
		else if (topologyError == 2)
			return "There is a branch with two tips"; // Should not normally
		// occur
		else if (topologyError == 3)
			return "The pink tip has more than one branch";
		else if (topologyError == 4)
			return "The pink node is isolated";
		else if (topologyError == 5)
			return "The pink inner node has only one branch";
		else if (topologyError == 6)
			return "The pink inner node has only two branches.\nThis is only permitted for the reference node/tree origin.";

		if (topologyError > 6) {
			if (topologyError == 7)
				return "The tree is disconnected";
			else {
				Branch.signalBranch = -1; // Ensure that a branch and a node are
				// not accentuated simultaneously
			}
			if (topologyError == 8)
				return "Either the topology is not tree-like: The pink node is on a cycle,\nor the look-ahead diameter is too high.";
		} else if (topologyError == 0) {
			Branch.signalBranch = -1; // No topology errors present - no need to
			// attenuate a node
			TreeNode.signalNode = -1;
			for (int i = 0; i < nodes.size(); i++)
				nodes.elementAt(i).setVisited(false);
			buildNewickStringFromTree();
		}
		return newickExpression;
	}

	public Object[] buildTreelikeLinkedList() {
		// First build a tree-like linked list based on the Vectors {branches}
		// and {nodes}
		Vector<Integer> theTipsBranches = new Vector<Integer>(); // Contains
		// branch
		// indices
		Vector<Integer> theInnerNodesBranches = new Vector<Integer>(); // Contains
		// branch
		// indices
		int innerNodesVisited = 0;

		for (int i = 0; i < nodes.size(); i++) {
			TreeNode treeNode = (TreeNode) nodes.elementAt(i);
			if (treeNode instanceof Tip) {
				// Collect all branches that start or end at treeNode
				theTipsBranches.clear(); // The branches are yet to find
				for (int j = 0; j < branches.size(); j++) {
					Branch branch = (Branch) branches.elementAt(j);
					branch.setVisited(true); // Mark the branch as "visited"

					if ((branch.getFirstNode().equals(treeNode))
							|| (branch.getSecondNode().equals(treeNode))) // replaced
					// by
					// equals
					{
						theTipsBranches.add(new Integer(j));
					}
				}

				// If no branch or more than one branch has been found,
				// the topology is incorrect; return an error code
				if (theTipsBranches.size() == 0) {
					Object[] o = { new Integer(1), treeNode };
					return o;
				} else if (theTipsBranches.size() > 1) {
					Branch tipBranch = branches.elementAt(theTipsBranches
							.elementAt(0));
					if ((tipBranch.getFirstNode() instanceof Tip)
							&& (tipBranch.getSecondNode() instanceof Tip)) {
						Object[] o = { new Integer(2), tipBranch };
						return o;
					} else {
						Object[] o = { new Integer(3), treeNode };
						return o;
					}
				} else // The Tip belongs to exactly one Branch:
				{
					// Add the second TreeNode of the Branch as neighbor to the
					// Tip
					Branch tipBranch = branches.elementAt(theTipsBranches
							.elementAt(0));
					if (tipBranch.getFirstNode().equals(treeNode)) // replaced
					// by equals
					{
						treeNode.addNeighbor(tipBranch.getSecondNode());
						tipBranch.getSecondNode().addNeighbor(treeNode);
					} else {
						treeNode.addNeighbor(tipBranch.getFirstNode());
						tipBranch.getFirstNode().addNeighbor(treeNode);
					}
				}
			} else if (treeNode instanceof InnerNode) {
				theInnerNodesBranches.clear(); // The branches are yet to find

				for (int j = 0; j < branches.size(); j++) {
					Branch branch = (Branch) branches.elementAt(j);
					if ((branch.getFirstNode().equals(treeNode))
							|| (branch.getSecondNode().equals(treeNode))) // replaced
					// by
					// equals
					{
						theInnerNodesBranches.add(new Integer(j));
					}
				}

				// If no branch or only one branch has been found,
				// the topology is incorrect; return an error code
				if (theInnerNodesBranches.size() == 0) {
					Object[] o = { new Integer(4), treeNode };
					return o;
				} else if (theInnerNodesBranches.size() == 1) {
					Object[] o = { new Integer(5), treeNode };
					return o;
				} else if (theInnerNodesBranches.size() == 2) // A definitive
				// candidate for
				// the location
				// of the root
				{
					if (wizard.referenceNodeId != treeNode.getId()) // If more
					// of them
					// exist,
					// the
					// topology
					// is wrong
					{
						// If the reference node is already set,
						// return an error message. Otherwise make
						// the current node the reference node
						if (wizard.referenceNodeId == -1)
							wizard.referenceNodeId = treeNode.id;
						{
							Object[] o = { new Integer(6), treeNode };
							return o;
						}
					}
				} else {
					// At least two branches have been found:
					// Add any branch not yet "visited" as neighbor of the node
					// under investigation
					for (int k = 0; k < theInnerNodesBranches.size(); k++) {
						Branch innerNodeBranch = (Branch) branches
								.elementAt((Integer) theInnerNodesBranches
										.elementAt(k));

						// One branch has been found: Add the second node of the
						// branch object
						// as a neighbor to the node under investigation
						if (innerNodeBranch.getFirstNode().equals(treeNode)) {
							treeNode.addNeighbor(innerNodeBranch
									.getSecondNode());
							innerNodeBranch.getSecondNode().addNeighbor(
									treeNode);
						} else {
							treeNode
									.addNeighbor(innerNodeBranch.getFirstNode());
							innerNodeBranch.getFirstNode()
									.addNeighbor(treeNode);
						}

						innerNodeBranch.setVisited(true); // Mark the branch as
						// "visited"
					}
				}
			}
		}

		// Return an error if the tree is not coherent:
		// It is sufficient to check whether all inner nodes are chained.
		// Collect a list with all inner nodes. Start with the first one and
		// visit all neighbors that are also inner nodes. Declare those nodes as
		// "visited".
		// Repeat until the whole list is traversed. If an inner node is not yet
		// visited,
		// the tree is not coherent.
		Vector<InnerNode> innerNodes = new Vector<InnerNode>();
		Stack<TreeNode> stack = new Stack<TreeNode>();
		for (int i = 0; i < nodes.size(); i++) {
			if (nodes.elementAt(i) instanceof InnerNode)
				innerNodes.add((InnerNode) nodes.elementAt(i));
		}
		stack.push(innerNodes.elementAt(0));

		do {
			TreeNode treeNode = (TreeNode) stack.pop();
			if (treeNode instanceof InnerNode) {
				treeNode.setVisited(true);
				innerNodesVisited++;

				Vector<TreeNode> neighbors = treeNode.getNeighbors();
				for (int i = 0; i < neighbors.size(); i++) {
					if (!neighbors.elementAt(i).getVisited())
						stack.push((TreeNode) neighbors.elementAt(i));
				}
			}
		} while (!stack.empty());

		if (innerNodes.size() > innerNodesVisited) {
			// At least one inner node is unreachable
			Object[] o = { new Integer(7), null };
			return o;
		}

		// Check whether the resulting topology contains a cycle

		for (int i = 0; i < nodes.size(); i++) {
			nodes.elementAt(i).flag = UNTOUCHED;
		}

		try {
			Iterator<InnerNode> innerNodesIt = innerNodes.iterator();
			while (innerNodesIt.hasNext()) {
				cycleFound = false;
				searchPath.clear(); // Clear current search path (which means no
				// node belongs to a cycle
				InnerNode startingNode = innerNodesIt.next();
				findCycles(startingNode, null); // Deposits the nodes that
				// belong to a cycle in
				// searchPath

				if (cycleFound) {
					Iterator<InnerNode> it = searchPath.iterator();
					boolean afterStartingNode = false;
					while (it.hasNext()) {
						InnerNode n = it.next();
						if (n == startingNode)
							afterStartingNode = true;
						if (afterStartingNode)
							System.out.println("" + n.getId());
					}
					Object[] o = { new Integer(8), null };
					return o;
				}
			}
		} catch (OutOfMemoryError e) {
		}
		;

		// The tree topology is correct
		Object[] o = new Object[2];
		o[0] = new Integer(0);
		o[1] = null;
		return o;
	} // buildTreeLikeLinkedList()

	private void buildNewickStringFromTree() {

		// Build up a hash with the branches;
		// one node serves as the key, the other as the value
		// branchesHash = new Hashtable<TreeNode, Branch>();

		InnerNode startNode = null;
		// Use the inner node the user has selected as a reference node for the
		// Newick String generation
		// Otherwise take the first inner node that has exactly two branches.
		// Otherwise take the first inner node stored.

		if (wizard.referenceNodeId != -1) // User has selected a reference node
		{
			for (int i = 0; i < nodes.size(); i++) {
				TreeNode node = nodes.elementAt(i);
				if (node instanceof InnerNode) {
					if (node.getId() == wizard.referenceNodeId) {
						startNode = (InnerNode) node;
						break;
					}
				}
			}
		} else // User has not selected a reference node
		{
			for (int i = 0; i < nodes.size(); i++) {
				TreeNode node = nodes.elementAt(i);
				if (node instanceof InnerNode) {
					if (node.getNeighbors().size() == 2) {
						startNode = (InnerNode) node;
						break;
					}
				}
			}

			if (startNode == null) {
				for (int i = 0; i < nodes.size(); i++) {
					TreeNode node = nodes.elementAt(i);
					if (node instanceof InnerNode) {
						startNode = (InnerNode) node;
						break;
					}
				}
			}
		}

		wizard.referenceNodeId = startNode.id;
		ndf = NumberUtility.getCurrentNumberFormat();
		if (startNode != null)
			newickExpression = traverseTree(startNode) + ";";
	} // buildNewickStringFromTree

	private String traverseTree(TreeNode treeNode) {
		String newickString = "";
		String nStr = "";
		treeNode.setVisited(true); // Ensure that the node is not touched again
		Vector<String[]> subtrees = new Vector<String[]>();
		String[] subtreeObj = null;

		if (treeNode instanceof InnerNode) {
			Vector<TreeNode> neighbors = ((InnerNode) treeNode).getNeighbors();

			subtreeObj = new String[2];

			newickString = "(";

			// Store the representations of all subtrees and
			// the lengths of the branches that link treeNode and the neighbors
			for (int i = 0; i < neighbors.size(); i++) {
				TreeNode neighbor = neighbors.elementAt(i);
				subtreeObj = new String[2];
				Branch b = topology.getBranchBetweenNodes(treeNode, neighbor);
				String l = Double.toString(b.getLength());

				if (!neighbor.getVisited()) {
					subtreeObj[0] = traverseTree(neighbor);
					subtreeObj[1] = l;
					subtrees.add(subtreeObj);
				}
			}

			// Build the Newick expression
			for (int i = 0; i <= subtrees.size() - 1; i++) {
				// Read out tree representation
				newickString += subtrees.elementAt(i)[0];
				// Read out respective branch length
				if (wizard.includeBranchLengths) {
					try {
						nStr = ndf.format((Double.parseDouble(subtrees
								.elementAt(i)[1])));
						newickString += ":" + nStr;
					} catch (Exception e) {
						return "";
					}
				}
				if (i < subtrees.size() - 1)
					newickString += ", ";
			}

			newickString += ")";
		} else if (treeNode instanceof Tip)
			newickString = ((Tip) treeNode).getTaxonName();
		return newickString;
	}

	public String getNewickString() {
		return newickExpression;
	};

	private void findCycles(TreeNode node, TreeNode predecessor) {
		if (!cycleFound) {
			if (node.flag == UNDERWAY) // Cycle found
			{
				// All nodes on current search path belong to the cycle
				cycleFound = true;
				TreeNode.signalNode = node.getId();
			} else if (node.flag == UNTOUCHED) {
				node.flag = UNDERWAY;
				if (node instanceof InnerNode)
					searchPath.add((InnerNode) node); // Prolongue the search
				// path with {node}
				// By referencing the predecessor from which all successors are
				// derived,
				// it is prevented that branches are examined in the opposite
				// direction
				Vector<TreeNode> neighbors = node.getNeighbors();
				Iterator<TreeNode> neighborsIt = neighbors.iterator();
				while (neighborsIt.hasNext()) {
					TreeNode neighbor = neighborsIt.next();
					if ((neighbor != predecessor)
							&& (neighbor instanceof InnerNode))
						findCycles(neighbor, node);
				}
				node.flag = FINISHED;
				// Remove {node] (the last element in Vector)
				// In a logical sense, it is the last element of current search
				// path
				searchPath.remove(searchPath.lastElement());
			}
		}
	}
}
