/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontFormatException;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Toolkit;

import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.awt.image.BufferedImage;

import java.io.IOException;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import com.apple.eawt.Application;
import com.apple.eawt.ApplicationAdapter;
import com.apple.eawt.ApplicationEvent;

import TreeSnatcher.GUI.FancyCursors;
import TreeSnatcher.GUI.GUIActions;
import TreeSnatcher.GUI.ImageBuffer;
import TreeSnatcher.GUI.ImagePanel;
import TreeSnatcher.GUI.InfiniteProgressPanel;
import TreeSnatcher.GUI.Magnifier;
import TreeSnatcher.GUI.ResultFrame;
import TreeSnatcher.GUI.StatusBar;
import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.GUI.AboutBox;
import TreeSnatcher.Utils.FileOperations;
import TreeSnatcher.Utils.NumberUtility;
import TreeSnatcher.Utils.Preferences;

public class MainWindow extends JFrame implements Constants {

	private static final long serialVersionUID = 1L;
	private JScrollPane scrollPane;
	private JPanel mainPanel;
	private JPanel wizardPanel;
	private ImagePanel imagePanel;
	private StatusBar statusBar;
	private Wizard wizard;
	private int screenWidth;
	private int screenHeight;
	private GUIActions guiActions;
	private ImageBuffer imageBuffer;
	private ImageOperations imageOperations;
	private FileOperations fileOperations;
	private Magnifier magnifier;
	private TreeTopology treeTopology;
	public InfiniteProgressPanel progressGlassPane;
	private ResultFrame newickPanel;
	private ActivityProtocol protocol;
	protected BufferedImage image;
	private String path;
	private MainWindow self = this;
	public static Font standardFont10, standardFont12, standardFont16;
	private String defaultTitle = "TreeSnatcher Plus: A Phylogenetic Tree Capturing Tool";

	public MainWindow(BufferedImage image, String path) {
		Toolkit toolkit = Toolkit.getDefaultToolkit();
		UIManager.put("OptionPane.questionIcon", tsIcon64);
		Image iconImage = toolkit.getImage(FancyCursors.class
				.getResource("/Icons/wood64.png"));
		this.setIconImage(iconImage);

		// MacOSX specific Application behavior
		// MacOSAboutHandler is only needed for MacOSX
		if (System.getProperty("mrj.version") != null)
			new MacOSAboutHandler();

		// Set localization
		Locale.setDefault(new Locale("us", "US"));

		this.image = image;
		this.path = path;
		initPreprocessing();
		initGUI();
		initUtilities();
		initTreeRecognition();
		initLogging();

		// Set references to objects that do not yet exist but are needed
		// elsewhere
		setMissingObjectReferences();
	}

	void initUtilities() {
		// Read the preferences file
		try {
			new Preferences();
		} catch (IOException e) {
		}
		;

		// Make available file operations
		fileOperations = new FileOperations(this, progressGlassPane,
				imageBuffer, wizard, statusBar);

		// Set number format
		new NumberUtility();
		NumberUtility.setDefaultNumberFormat();

		// Use the font from the file "Helvetica.dfont"
		Font font = null;
		try {
			font = Font.createFont(Font.TRUETYPE_FONT, Preferences.fontFile);
		} catch (FontFormatException e) {
		} catch (IOException e) {
		}
		GraphicsEnvironment ge = GraphicsEnvironment
				.getLocalGraphicsEnvironment();
		if (font != null) {
			ge.registerFont(font);
			standardFont10 = font.deriveFont(10f);
			standardFont12 = font.deriveFont(12f);
		} else {
			standardFont10 = new Font("SansSerif", Font.PLAIN, 10);
			standardFont12 = new Font("SansSerif", Font.PLAIN, 12);
		}

	}

	void initGUI() {
		// Learn the dimension of the Desktop
		Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
		screenWidth = (int) screen.getWidth();
		screenHeight = (int) screen.getHeight();

		// Build a status bar
		JPanel statusPanel = new JPanel(new BorderLayout());
		statusBar = new StatusBar(this, statusPanel);

		// Add the frame in which the Phylip-Expression (Newick String) gets
		// displayed
		newickPanel = new ResultFrame(this, scrollPane);

		// Build a scroll pane that will contain imageLabel as the view client
		// (the image)
		scrollPane = new JScrollPane();
		scrollPane.setMaximumSize(new Dimension(screenWidth - 150,
				screenHeight - 100));
		scrollPane.setPreferredSize(new Dimension(814, 440));
		scrollPane.setMinimumSize(new Dimension(300, 300));
		scrollPane.getViewport().setBackground(Color.white);
		scrollPane.setWheelScrollingEnabled(true);

		// Build the wizard that guides the user through the program
		wizardPanel = new JPanel();
		wizard = new Wizard(this, wizardPanel, scrollPane);
		wizard.programPath = path;

		// Initialize custom mouse cursors
		FancyCursors cursors = new FancyCursors(wizard);
		scrollPane.setCursor(cursors.getHandCursor());

		// Build a new magnifier
		magnifier = new Magnifier(8, imageBuffer, wizard);

		// Build a custom glass pane
		progressGlassPane = new InfiniteProgressPanel();
		setGlassPane(progressGlassPane);

		// Central user interactivity class:
		// Handle all mouse operations and conduct all modifications to the
		// image
		imagePanel = new ImagePanel(this, image, scrollPane, statusBar,
				magnifier, wizard, newickPanel, imageBuffer, imageOperations,
				progressGlassPane, cursors, fileOperations);
		// Add custom menues to the screen menu bar
		// Add a popup menu
		// Add a tool bar to the main window
		// Build controls and actions
		guiActions = new GUIActions(this, imagePanel, imageBuffer, magnifier,
				fileOperations, wizard);

		// Build a main panel for the image and the Wizard/Phylip dialog
		mainPanel = new JPanel();
		mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.Y_AXIS));
		mainPanel.add(scrollPane);
		mainPanel.add(wizardPanel);
		mainPanel.setBorder(BorderFactory.createLoweredBevelBorder());

		// Add all controls to this window's content pane
		this.getContentPane().add(mainPanel, BorderLayout.CENTER);
		this.getContentPane().add(statusPanel, BorderLayout.SOUTH);
		this.setLocation(0, 20);
		this.setTitle(defaultTitle);
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		this.addWindowListener(new WindowClosingAdapter(true));
		this.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
		this.pack();
	}

	void initPreprocessing() {
		// Instantiate a buffer class that stores a copy of the source image
		// A copy is needed for Undo operations and further reference
		imageBuffer = new ImageBuffer(image);

		// Make available all image operations
		imageOperations = new ImageOperations(wizard, imageBuffer);
	}

	void initTreeRecognition() {
		// Make available all operations that refer to the tree topology
		// recognition
		treeTopology = new TreeTopology(imageOperations, imagePanel,
				imageBuffer, wizard, this, guiActions);
	}

	void initLogging() {
		new ActivityConstraints(imagePanel, treeTopology, wizard);
		protocol = new ActivityProtocol(imagePanel, imageBuffer,
				imageOperations, wizard, guiActions, treeTopology);
	}

	public void swapWizardAndNewickPanels() {
		if (wizard.showResultFrame) {
			if (wizardPanel != null) {
				wizardPanel.setVisible(false);
				mainPanel.remove(wizardPanel);
			}
			;
			if (newickPanel != null) {
				mainPanel.add(newickPanel);
				newickPanel.setVisible(true);
			}
			;
		} else {
			if (newickPanel != null) {
				newickPanel.setVisible(false);
				mainPanel.remove(newickPanel);
			}
			;
			if ((wizardPanel != null) && (wizard.useWizard)) {
				mainPanel.add(wizardPanel);
				wizardPanel.setVisible(true);
			}
			;
		}

		invalidate();
		mainPanel.validate();
	}

	void setMissingObjectReferences() {
		imageOperations.setObjectReferences(wizard, treeTopology);
		magnifier.setObjectReferences(imageBuffer, imagePanel, wizard,
				treeTopology);
		imagePanel.setObjectReferences(this, imageBuffer, imageOperations,
				guiActions, treeTopology, protocol);
		imageBuffer
				.setObjectReferences(guiActions, imagePanel, imageOperations);
		wizard.setObjectReferences(guiActions);
		guiActions
				.setObjectReferences(imagePanel, fileOperations, treeTopology);
		newickPanel.setObjectReferences(guiActions, fileOperations,
				treeTopology);
		fileOperations.setObjectReferences(treeTopology, imagePanel,
				imageOperations);
		newickPanel.setObjectReferences(wizard);
	}

	// MacOSX specific stuff
	class MacOSAboutHandler extends Application {
		public MacOSAboutHandler() {
			addApplicationListener(new AboutBoxHandler());
		}

		class AboutBoxHandler extends ApplicationAdapter {
			public void handleAbout(ApplicationEvent event) {
				event.setHandled(true);
				new AboutBox(new JFrame()).setVisible(true);
			}

			public void handleQuit(ApplicationEvent event) {
				boolean handled = showQuitDialogBox();
				event.setHandled(handled);
				if (handled)
					protocol.purgeData();
				else
					self.setVisible(true);
			}
		}
	}

	public void changeTitle(String fileName) {
		this.setTitle(fileName);
	}

	private boolean showQuitDialogBox() {
		// There is something that should be saved
		// Display a modal dialog box

		if (!Wizard.undoHistoryEmptyAfterLastSave) {
			JFrame frame = new JFrame();
			frame.setLocation(screenWidth / 2, (int) screenHeight / 2);

			Object[] options = { "Don't quit", "Quit anyway" };
			int n = JOptionPane
					.showOptionDialog(
							frame,
							"Your work is not yet saved.\nWould you like to quit without saving?",
							"", JOptionPane.YES_NO_OPTION,
							JOptionPane.QUESTION_MESSAGE, tsIcon64, options, // the
							// titles
							// of
							// buttons
							options[0]); // default button title
			if (n == JOptionPane.YES_OPTION)
				return false;
			else if (n == JOptionPane.NO_OPTION)
				return true;
			else if (n == JOptionPane.CLOSED_OPTION)
				return false;
		}
		return true; // {true} quits application
	}

	public void setWizard(Wizard w) {
		this.wizard = w;
	};

	public Wizard getWizard() {
		return this.wizard;
	};

	class WindowClosingAdapter extends WindowAdapter {
		private boolean exitSystem;

		public WindowClosingAdapter(boolean exitSystem) {
			this.exitSystem = exitSystem;
		}

		public WindowClosingAdapter() {
			this(false);
		};

		public void windowClosing(WindowEvent e) {
			boolean handled = showQuitDialogBox();

			if (handled) {
				protocol.purgeData();
				e.getWindow().setVisible(false);
				e.getWindow().dispose();
				if (exitSystem)
					System.exit(0);
			} else
				self.setVisible(true);
		}

	}

}
