/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Shape;
import java.awt.Transparency;
import java.awt.geom.Area;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;
import java.util.Stack;
import java.util.Vector;

import TreeSnatcher.GUI.ImageBuffer;
import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.GUI.Pixel;
import TreeSnatcher.Utils.Quantize;
import TreeSnatcher.Core.TreeTopology;

public class ImageOperations implements Constants {
	public ImageOperations(Wizard wizard, ImageBuffer imageBuffer) {
		this.wizard = wizard;
		this.imageBuffer = imageBuffer;
	}

	private int[] histogram;
	private Wizard wizard;
	private ImageBuffer imageBuffer;
	private TreeTopology topology;

	// This method converts an 8-bit RGB image to greyscale
	// For the weighting of colors red, green and blue it uses the values
	// proposed by recommendation ITU-BT.709
	// "International Telecommunications Union", 1998
	// The method also modifies the alpha values of the image
	public BufferedImage convertToGreyscale(BufferedImage image,
			Shape selection, double[] factors) throws OutOfMemoryError {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();
		int rgb = 0;
		int c = 0, a = 0, r = 0, g = 0, b = 0;
		int lum = 0, lum_pre = 0;

		for (int i = xb; i < xb + w; i++) {
			for (int j = yb; j < yb + h; j++) {
				rgb = image.getRGB(i, j);
				r = (rgb & 0xff0000) >> 16;
				g = (rgb & 0x00ff00) >> 8;
				b = (rgb & 0x0000ff);

				{
					lum_pre = (int) Math.round(r * factors[0] + g * factors[1]
							+ b * factors[2]);
					lum = Math.max(0, Math.min(255, lum_pre));
					c = new Color(lum, lum, lum).getRGB();
				}

				image.setRGB(i, j, c);
			}
		}
		return image;
	}

	// Histogram Stretch with Saturation on a grayscale image
	// Determine minimum and maximum among the shades in the image
	// Ignore left and right stripes that represent 0,5% of the shades
	public BufferedImage stretchHistogram(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();
		int c, s, rgb, newShade, cnt, low = 0, high = 0;

		// Count all bins in the histogram that are not zero
		cnt = 0;
		for (int i = 0; i < 255; i++)
			if (histogram[i] != 0)
				cnt++;
		int f = (int) Math.round(0.05 * cnt);
		// Set minimum and maximum shades that get stretched;
		// All shades below them are projected to 0 and 255 respectively
		int min = f, max = 255 - f;
		boolean minimumSet = false, maximumSet = false;

		for (int shade = 0; shade < 255; shade++) {
			if (minimumSet == false) {
				if (histogram[shade] != 0) {
					low = shade;
					minimumSet = true;
				}
				;
			}
		}

		for (int shade = 255; shade > 0; shade--) {
			if (maximumSet == false) {
				if (histogram[shade] != 0) {
					high = shade;
					maximumSet = true;
				}
				;
			}
		}

		for (int i = xb; i < xb + w; i++) {
			for (int j = yb; j < yb + h; j++) {
				rgb = image.getRGB(i, j);
				c = new Color(rgb).getRed();
				s = 255 * (c - low) / (high - low);
				if (s < min)
					s = 0;
				if (s > max)
					s = 255;
				newShade = new Color(s, s, s).getRGB();
				image.setRGB(i, j, newShade);
			}
		}
		return image;
	}

	public BufferedImage convertARGBToRGB(BufferedImage image)
			throws OutOfMemoryError {
		BufferedImage newImage;
		int width = image.getWidth();
		int height = image.getHeight();
		if (image.getTransparency() == Transparency.OPAQUE) {
			newImage = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_RGB);
			return newImage;
		} else if (!image.isAlphaPremultiplied()) // The RGB channels are not
		// premultiplied
		{
			newImage = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_RGB);
			float rc = 0, gc = 0, bc = 0;
			int t = new Color(defaultTransparencyShade).getRed();

			Raster raster = image.getRaster();
			int[] redSamples = raster.getSamples(0, 0, width, height, 0,
					(int[]) null);
			int[] greenSamples = raster.getSamples(0, 0, width, height, 1,
					(int[]) null);
			int[] blueSamples = raster.getSamples(0, 0, width, height, 2,
					(int[]) null);
			int[] alphaSamples = raster.getSamples(0, 0, width, height, 3,
					(int[]) null);

			for (int x = 0; x < width; x++) {
				for (int y = 0; y < height; y++) {
					int r = redSamples[y * width + x];
					int g = greenSamples[y * width + x];
					int b = blueSamples[y * width + x];
					int a = alphaSamples[y * width + x];
					float f = a * 1.0f / 255.0f;

					if (f == 0.0f) {
						rc = t;
						gc = t;
						bc = t;
					} else if (a == 1.0f) {
						rc = r;
						gc = g;
						bc = b;
					} else {
						rc = Math.max(0, Math.min(255, r * f + (1 - f) * t));
						gc = Math.max(0, Math.min(255, g * f + (1 - f) * t));
						bc = Math.max(0, Math.min(255, b * f + (1 - f) * t));
					}
					newImage.setRGB(x, y, new Color((int) rc, (int) gc,
							(int) bc).getRGB());
				}
			}

			return newImage;
		} else // NOT USED: The RGB channels have been premultipied with their
		// respective alpha values
		{
			newImage = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_RGB);
			int rc, gc, bc;

			for (int x = 0; x < width; x++) {
				for (int y = 0; y < height; y++) {
					Color v = new Color(image.getRGB(x, y));
					float ar = 1 / v.getAlpha();
					int r = v.getRed();
					int g = v.getGreen();
					int b = v.getBlue();
					rc = r;
					gc = g;
					bc = b;
					if (ar != 0) {
						rc = (int) (r * ar);
						gc = (int) (g * ar);
						bc = (int) (b * ar);
					}
				}
			}
			return newImage;
		}
	}

	// Inverts an RGB color image
	public BufferedImage invertImage(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		for (int i = xb; i < w + xb; i++) {
			for (int j = yb; j < h + yb; j++) {
				int lum = image.getRGB(i, j);
				Color col = new Color(lum);
				int red = col.getRed();
				int green = col.getGreen();
				int blue = col.getBlue();
				image.setRGB(i, j,
						new Color(255 - red, 255 - green, 255 - blue).getRGB());
			}
		}
		return image;
	}

	// Binarizes an RGB color image
	// Uses the corresponding luminance value for an RGB color triple instead of
	// the color components
	// For the weighting of colors red, green and blue it uses the values
	// proposed by recommendation ITU-BT.709
	// "International Telecommunications Union", 1998
	public BufferedImage binarizeImage(BufferedImage image, Shape selection,
			int threshold) throws OutOfMemoryError {
		double[] f = defaultGreyscaleFactors;
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();
		Color rgb;
		int shade;

		for (int x = xb; x < xb + w; x++) {
			for (int y = yb; y < yb + h; y++) {
				rgb = new Color(image.getRGB(x, y));
				shade = (int) ((int) f[0] * rgb.getRed() + f[1]
						* rgb.getGreen() + f[2] * rgb.getBlue());
				if (shade < threshold)
					shade = BIN1;
				if (shade >= threshold)
					shade = BIN0;
				image.setRGB(x, y, shade);
			}
		}
		return image;
	}

	// Smoothes the color bands of an RGB image separately
	public BufferedImage smoothImage(BufferedImage image, Shape selection,
			int kernelWidth) throws OutOfMemoryError {
		double sigma = (double) kernelWidth / 2.0;
		double alpha = 0.5 / (sigma * sigma);
		double[] mask = new double[2 * kernelWidth + 1];

		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		// Calculate Gauss filter mask
		for (int x = 0; x < sigma; x++) {
			mask[x] = Math.exp(-alpha * x * x);
			mask[x + kernelWidth] = Math.exp(-alpha * x * x);
		}

		int[] rgbTriplets = image.getRGB(xb, yb, w, h, null, 0, w);
		int[] temp = new int[w * h];
		int[] samples = new int[w * h];
		int[] redOut = new int[w * h];
		int[] greenOut = new int[w * h];
		int[] blueOut = new int[w * h];

		// Red band
		for (int i = 0; i < w * h; i++)
			samples[i] = new Color(rgbTriplets[i]).getRed();

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				double sum = 0;
				double weight = 0;
				for (int i = x; i <= x + 2 * kernelWidth; i++) {
					if (((i - kernelWidth) < 0) || ((i - kernelWidth) >= w))
						continue;
					sum = sum + samples[y * w + (i - kernelWidth)]
							* mask[i - x];
					weight = weight + mask[i - x];
				}
				redOut[y * w + x] = (int) (sum / weight);
			}
		}

		temp = samples;
		samples = redOut;
		redOut = temp;

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				double sum = 0;
				double weight = 0;
				for (int i = y; i <= y + 2 * kernelWidth; i++) {
					if (((i - kernelWidth) < 0) || ((i - kernelWidth) >= h))
						continue;
					sum = sum + samples[(i - kernelWidth) * w + x]
							* mask[i - y];
					weight = weight + mask[i - y];
				}
				redOut[y * w + x] = (int) (sum / weight);
			}
		}

		// Green band
		for (int i = 0; i < w * h; i++)
			samples[i] = new Color(rgbTriplets[i]).getGreen();

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				double sum = 0;
				double weight = 0;
				for (int i = x; i <= x + 2 * kernelWidth; i++) {
					if (((i - kernelWidth) < 0) || ((i - kernelWidth) >= w))
						continue;
					sum = sum + samples[y * w + (i - kernelWidth)]
							* mask[i - x];
					weight = weight + mask[i - x];
				}
				greenOut[y * w + x] = (int) (sum / weight);
			}
		}

		temp = samples;
		samples = greenOut;
		greenOut = temp;

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				double sum = 0;
				double weight = 0;
				for (int i = y; i <= y + 2 * kernelWidth; i++) {
					if (((i - kernelWidth) < 0) || ((i - kernelWidth) >= h))
						continue;
					sum = sum + samples[(i - kernelWidth) * w + x]
							* mask[i - y];
					weight = weight + mask[i - y];
				}
				greenOut[y * w + x] = (int) (sum / weight);
			}
		}

		// Blue band
		for (int i = 0; i < w * h; i++)
			samples[i] = new Color(rgbTriplets[i]).getBlue();

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				double sum = 0;
				double weight = 0;
				for (int i = x; i <= x + 2 * kernelWidth; i++) {
					if (((i - kernelWidth) < 0) || ((i - kernelWidth) >= w))
						continue;
					sum = sum + samples[y * w + (i - kernelWidth)]
							* mask[i - x];
					weight = weight + mask[i - x];
				}
				blueOut[y * w + x] = (int) (sum / weight);
			}
		}

		temp = samples;
		samples = blueOut;
		blueOut = temp;

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				double sum = 0;
				double weight = 0;
				for (int i = y; i <= y + 2 * kernelWidth; i++) {
					if (((i - kernelWidth) < 0) || ((i - kernelWidth) >= h))
						continue;
					sum = sum + samples[(i - kernelWidth) * w + x]
							* mask[i - y];
					weight = weight + mask[i - y];
				}
				blueOut[y * w + x] = (int) (sum / weight);
			}
		}

		int r, g, b;
		for (int i = 0; i < w * h; i++) {
			r = Math.min(255, Math.max(0, (int) redOut[i]));
			g = Math.min(255, Math.max(0, (int) greenOut[i]));
			b = Math.min(255, Math.max(0, (int) blueOut[i]));
			rgbTriplets[i] = new Color(r, g, b).getRGB();
		}

		BufferedImage newImage = new BufferedImage(image.getColorModel(), image
				.getRaster(), image.isAlphaPremultiplied(), null);
		newImage.setRGB(xb, yb, w, h, rgbTriplets, 0, w);

		rgbTriplets = null;
		temp = null;
		samples = null;
		redOut = null;
		greenOut = null;
		blueOut = null;

		return newImage;
	}

	public BufferedImage sharpenUSM(BufferedImage image, Shape selection,
			int kernelWidth, double a) throws OutOfMemoryError {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();
		int[] rgbTriples = image.getRGB(xb, yb, w, h, null, 0, w);
		BufferedImage smoothedImage = smoothImage(image, selection, kernelWidth);
		int[] rgbSmooth = smoothedImage.getRGB(xb, yb, w, h, null, 0, w);
		int[] mask = new int[w * h];
		int[] srcSamples = new int[w * h];
		int[] smoSamples = new int[w * h];

		// Red band
		for (int i = 0; i < w * h; i++)
			srcSamples[i] = new Color(rgbTriples[i]).getRed();
		for (int i = 0; i < w * h; i++)
			smoSamples[i] = new Color(rgbSmooth[i]).getRed();

		// Mask: Subtract a Gauss-filtered version of the original image from
		// the image
		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				mask[y * w + x] = srcSamples[y * w + x] - smoSamples[y * w + x];
			}
		}

		int[] newRedSamples = new int[w * h];
		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				newRedSamples[y * w + x] = srcSamples[y * w + x] + (int) a
						* mask[y * w + x];
			}
		}

		// Green band
		for (int i = 0; i < w * h; i++)
			srcSamples[i] = new Color(rgbTriples[i]).getGreen();
		for (int i = 0; i < w * h; i++)
			smoSamples[i] = new Color(rgbSmooth[i]).getGreen();

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				mask[y * w + x] = srcSamples[y * w + x] - smoSamples[y * w + x];
			}
		}

		int[] newGreenSamples = new int[w * h];
		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				newGreenSamples[y * w + x] = srcSamples[y * w + x] + (int) a
						* mask[y * w + x];
			}
		}

		// Blue band
		for (int i = 0; i < w * h; i++)
			srcSamples[i] = new Color(rgbTriples[i]).getBlue();
		for (int i = 0; i < w * h; i++)
			smoSamples[i] = new Color(rgbSmooth[i]).getBlue();

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				mask[y * w + x] = srcSamples[y * w + x] - smoSamples[y * w + x];
			}
		}

		smoothedImage = null;
		rgbSmooth = null;
		smoSamples = null;

		int[] newBlueSamples = new int[w * h];
		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				newBlueSamples[y * w + x] = srcSamples[y * w + x] + (int) a
						* mask[y * w + x];
			}
		}

		int r, g, b;
		for (int i = 0; i < w * h; i++) {
			r = Math.min(255, Math.max(0, (int) newRedSamples[i]));
			g = Math.min(255, Math.max(0, (int) newGreenSamples[i]));
			b = Math.min(255, Math.max(0, (int) newBlueSamples[i]));
			rgbTriples[i] = new Color(r, g, b).getRGB();
		}

		// NewImage: Add Mask using weight factor a to the original image
		BufferedImage newImage = new BufferedImage(image.getColorModel(), image
				.getRaster(), image.isAlphaPremultiplied(), null);
		newImage.setRGB(xb, yb, w, h, rgbTriples, 0, w);
		newRedSamples = null;
		newGreenSamples = null;
		newBlueSamples = null;

		return newImage;
	}

	// Uses a median filter to despeckle separately each color band of an RGB
	// image
	public BufferedImage despeckleImage(BufferedImage image, Shape selection,
			int rectWidth) throws OutOfMemoryError {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		int[] hist = new int[256];
		int area = rectWidth * rectWidth;
		rectWidth = rectWidth / 2;
		int[] rgbTriplets = image.getRGB(xb, yb, w, h, null, 0, w);
		int[] redSamples = new int[w * h];
		int[] redTemp = new int[w * h];
		int[] greenTemp = new int[w * h];
		int[] blueTemp = new int[w * h];

		// Process red band
		for (int i = 0; i < w * h; i++)
			redSamples[i] = new Color(rgbTriplets[i]).getRed();

		for (int y = 0; y < h; y++) {
			int ymin = Math.max(0, y - rectWidth);
			int ymax = Math.min(h - 1, y + rectWidth);

			for (int x = 0; x < w; x++) {
				int xmin = Math.max(0, x - rectWidth);
				int xmax = Math.min(w - 1, x + rectWidth);
				int sum = 0;
				int i;

				for (i = 0; i < 255; i++)
					hist[i] = 0;
				for (int yy = ymin; yy <= ymax; yy++) {
					for (int xx = xmin; xx <= xmax; xx++)
						hist[redSamples[yy * w + xx]] += 1;
				}
				i = -1;
				while ((sum < area / 2) && (i < 255))
					sum = sum + hist[++i];
				redTemp[x + y * w] = i;
			}
		}

		// Process green band
		for (int i = 0; i < w * h; i++)
			redSamples[i] = new Color(rgbTriplets[i]).getGreen();

		for (int y = 0; y < h; y++) {
			int ymin = Math.max(0, y - rectWidth);
			int ymax = Math.min(h - 1, y + rectWidth);

			for (int x = 0; x < w; x++) {
				int xmin = Math.max(0, x - rectWidth);
				int xmax = Math.min(w - 1, x + rectWidth);
				int sum = 0;
				int i;

				for (i = 0; i < 255; i++)
					hist[i] = 0;
				for (int yy = ymin; yy <= ymax; yy++) {
					for (int xx = xmin; xx <= xmax; xx++)
						hist[redSamples[yy * w + xx]] += 1;
				}
				i = -1;
				while ((sum < area / 2) && (i < 255))
					sum = sum + hist[++i];
				greenTemp[x + y * w] = i;
			}
		}

		// Process blue band
		for (int i = 0; i < w * h; i++)
			redSamples[i] = new Color(rgbTriplets[i]).getBlue();

		for (int y = 0; y < h; y++) {
			int ymin = Math.max(0, y - rectWidth);
			int ymax = Math.min(h - 1, y + rectWidth);

			for (int x = 0; x < w; x++) {
				int xmin = Math.max(0, x - rectWidth);
				int xmax = Math.min(w - 1, x + rectWidth);
				int sum = 0;
				int i;

				for (i = 0; i < 255; i++)
					hist[i] = 0;
				for (int yy = ymin; yy <= ymax; yy++) {
					for (int xx = xmin; xx <= xmax; xx++)
						hist[redSamples[yy * w + xx]] += 1;
				}
				i = -1;
				while ((sum < area / 2) && (i < 255))
					sum = sum + hist[++i];
				blueTemp[x + y * w] = i;
			}
		}

		int r, g, b;
		for (int i = 0; i < w * h; i++) {
			r = Math.min(255, Math.max(0, (int) redTemp[i]));
			g = Math.min(255, Math.max(0, (int) greenTemp[i]));
			b = Math.min(255, Math.max(0, (int) blueTemp[i]));
			rgbTriplets[i] = new Color(r, g, b).getRGB();
		}

		BufferedImage newImage = new BufferedImage(image.getColorModel(), image
				.getRaster(), image.isAlphaPremultiplied(), null);
		newImage.setRGB(xb, yb, w, h, rgbTriplets, 0, w);

		return newImage;
	}

	public void copyIntoImage(BufferedImage image, BufferedImage subimage,
			int x, int y, Shape clippingArea) {
		if (subimage != null) {
			Graphics2D g2 = (Graphics2D) image.getGraphics();
			g2.setClip(clippingArea);
			g2.drawImage(subimage, x, y, null);
		}
	}

	public void copyIntoImage(BufferedImage subimage, int x, int y) {
		if (subimage != null) {
			BufferedImage image = imageBuffer.getProcessedImage();
			Graphics2D g2 = (Graphics2D) image.getGraphics();
			g2.drawImage(subimage, x, y, null);
		}
	}

	// Brightens/darkens an RGB color image
	public BufferedImage brightenImage(BufferedImage image, Shape selection,
			double factor, int offset) throws OutOfMemoryError {

		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();
		int r, g, b;
		Color rgb;

		for (int i = xb; i < w + xb; i++) {
			for (int j = yb; j < h + yb; j++) {
				rgb = new Color(image.getRGB(i, j));
				r = (int) Math.max(0, Math.min(255,
						((rgb.getRed() + offset) * factor)));
				g = (int) Math.max(0, Math.min(255,
						((rgb.getGreen() + offset) * factor)));
				b = (int) Math.max(0, Math.min(255,
						((rgb.getBlue() + offset) * factor)));
				image.setRGB(i, j, new Color(r, g, b).getRGB());
			}
		}

		return image;
	}

	// Performs locally adaptive minimum/maximum filtering separately on each
	// color band of an RGB image
	public BufferedImage locAdaptMinMax(BufferedImage image, Shape selection,
			int rectWidth) throws OutOfMemoryError {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		int[] hist = new int[256];
		int vmin = 256, vmax = -1;
		rectWidth = rectWidth / 2;

		int[] rgbTriplets = image.getRGB(xb, yb, w, h, null, 0, w);
		int[] redSamples = new int[w * h];
		int[] redTemp = new int[w * h];
		int[] greenTemp = new int[w * h];
		int[] blueTemp = new int[w * h];

		// Red color component
		for (int i = 0; i < w * h; i++)
			redSamples[i] = new Color(rgbTriplets[i]).getRed();

		for (int y = 0; y < h; y++) {
			int ymin = Math.max(0, y - rectWidth);
			int ymax = Math.min(h - 1, y + rectWidth);

			for (int x = 0; x < w; x++) {
				int xmin = Math.max(0, x - rectWidth);
				int xmax = Math.min(w - 1, x + rectWidth);
				int i;
				int v = 0;

				for (i = 0; i < 255; i++)
					hist[i] = 0;
				vmin = 256;
				vmax = -1;
				for (int yy = ymin; yy <= ymax; yy++) {
					for (int xx = xmin; xx <= xmax; xx++) {
						v = redSamples[yy * w + xx];
						if (v < vmin)
							vmin = v;
						if (v > vmax)
							vmax = v;
					}
				}
				if (Math.abs(redSamples[x + y * w] - vmin) < Math
						.abs(redSamples[x + y * w] - vmax))
					redTemp[x + y * w] = vmin;
				else
					redTemp[x + y * w] = vmax;
			}
		}

		// Green color component
		for (int i = 0; i < w * h; i++)
			redSamples[i] = new Color(rgbTriplets[i]).getGreen();

		for (int y = 0; y < h; y++) {
			int ymin = Math.max(0, y - rectWidth);
			int ymax = Math.min(h - 1, y + rectWidth);

			for (int x = 0; x < w; x++) {
				int xmin = Math.max(0, x - rectWidth);
				int xmax = Math.min(w - 1, x + rectWidth);
				int i;
				int v = 0;

				for (i = 0; i < 255; i++)
					hist[i] = 0;
				vmin = 256;
				vmax = -1;
				for (int yy = ymin; yy <= ymax; yy++) {
					for (int xx = xmin; xx <= xmax; xx++) {
						v = redSamples[yy * w + xx];
						if (v < vmin)
							vmin = v;
						if (v > vmax)
							vmax = v;
					}
				}
				if (Math.abs(redSamples[x + y * w] - vmin) < Math
						.abs(redSamples[x + y * w] - vmax))
					greenTemp[x + y * w] = vmin;
				else
					greenTemp[x + y * w] = vmax;
			}
		}

		// Blue color component
		for (int i = 0; i < w * h; i++)
			redSamples[i] = new Color(rgbTriplets[i]).getBlue();

		for (int y = 0; y < h; y++) {
			int ymin = Math.max(0, y - rectWidth);
			int ymax = Math.min(h - 1, y + rectWidth);

			for (int x = 0; x < w; x++) {
				int xmin = Math.max(0, x - rectWidth);
				int xmax = Math.min(w - 1, x + rectWidth);
				int i;
				int v = 0;

				for (i = 0; i < 255; i++)
					hist[i] = 0;
				vmin = 256;
				vmax = -1;
				for (int yy = ymin; yy <= ymax; yy++) {
					for (int xx = xmin; xx <= xmax; xx++) {
						v = redSamples[yy * w + xx];
						if (v < vmin)
							vmin = v;
						if (v > vmax)
							vmax = v;
					}
				}
				if (Math.abs(redSamples[x + y * w] - vmin) < Math
						.abs(redSamples[x + y * w] - vmax))
					blueTemp[x + y * w] = vmin;
				else
					blueTemp[x + y * w] = vmax;
			}
		}

		int r, g, b;
		for (int i = 0; i < w * h; i++) {
			r = Math.min(255, Math.max(0, (int) redTemp[i]));
			g = Math.min(255, Math.max(0, (int) greenTemp[i]));
			b = Math.min(255, Math.max(0, (int) blueTemp[i]));
			rgbTriplets[i] = new Color(r, g, b).getRGB();
		}

		BufferedImage newImage = new BufferedImage(image.getColorModel(), image
				.getRaster(), image.isAlphaPremultiplied(), null);
		newImage.setRGB(xb, yb, w, h, rgbTriplets, 0, w);

		rgbTriplets = null;
		redSamples = null;
		redTemp = null;
		greenTemp = null;
		blueTemp = null;

		return newImage;
	}

	// Performs local binarization of an image portion using a separated Gauss
	// filter
	// Modified from Aurich, Volker: Lecture Bildverarbeitung I, HHU
	// Duesseldorf, Germany, 2004
	public BufferedImage autoGaussBinarization(BufferedImage image,
			Shape selection, double sigma, double factor)
			throws OutOfMemoryError {
		int kernelWidth = (int) (2 * sigma);
		double alpha = 0.5 / (sigma * sigma);
		double[] mask = new double[2 * kernelWidth + 1];

		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		Color c;

		// Calculate Gauss filter mask
		for (int x = 0; x <= kernelWidth; x++) {
			mask[x] = Math.exp(-alpha * x * x);
			mask[x + kernelWidth] = mask[x];
		}

		int[] rgbTriplets = image.getRGB(xb, yb, w, h, null, 0, w);
		int[] in = new int[w * h];
		int[] temp = new int[w * h];
		int[] out = new int[w * h];

		for (int i = 0; i < w * h; i++) {
			c = new Color(rgbTriplets[i]);

			in[i] = c.getRed();
		}

		for (int y = 0; y < h; y++) {
			for (int x = 0; x < w; x++) {
				double sum = 0;
				double weight = 0;
				for (int i = x - kernelWidth; i <= x + kernelWidth; i++) {
					if ((i < 0) || (i >= w))
						continue;
					sum = sum + in[y * w + i] * mask[i - x + kernelWidth];
					weight = weight + mask[i - x + kernelWidth];
				}
				temp[y * w + x] = (int) (sum / weight);
			}
		}

		for (int x = 0; x < w; x++) {
			for (int y = 0; y < h; y++) {
				double sum = 0;
				double weight = 0;
				for (int i = y - kernelWidth; i <= y + kernelWidth; i++) {
					if ((i < 0) || (i >= h))
						continue;
					sum = sum + temp[i * w + x] * mask[i - y + kernelWidth];
					weight = weight + mask[i - y + kernelWidth];
				}
				if (in[y * w + x] >= factor * sum / weight)
					out[y * w + x] = 255;
				else
					out[y * w + x] = 0;
			}
		}

		int s;
		for (int l = 0; l < w * h; l++) {
			s = out[l];
			rgbTriplets[l] = new Color(s, s, s).getRGB();
		}

		BufferedImage newImage = new BufferedImage(image.getColorModel(), image
				.getRaster(), image.isAlphaPremultiplied(), null);
		newImage.setRGB(xb, yb, w, h, rgbTriplets, 0, w);

		return newImage;
	}

	// The following Thinning algorithm is modified and extended for
	// image portions from Wayne Rasband's ImageJ package
	public BufferedImage skeletonizeImage(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		/**
		 * Uses a lookup table to repeatably remove pixels from the edges of
		 * objects in a binary image, reducing them to single pixel wide
		 * skeletons. Based on an a thinning algorithm by by Zhang and Suen
		 * (CACM, March 1984, 236-239). There is an entry in the table for each
		 * of the 256 possible 3x3 neighborhood configurations. An entry of '1'
		 * means delete pixel on first pass, '2' means delete pixel on second
		 * pass, and '3' means delete on either pass. A graphical representation
		 * of the 256 neighborhoods indexed by the table is available at
		 * "http://rsb.info.nih.gov/ij/images/skeletonize-table.gif".
		 */

		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int width = (int) bounds.getWidth();
		int height = (int) bounds.getHeight();
		Graphics2D g = (Graphics2D) image.getGraphics();

		// Fabricate a larger image of the source image with a white
		// (background) one-pixel-border
		// Needed for the skeletonization of the border pixels
		BufferedImage temp = new BufferedImage(width + 2, height + 2,
				BufferedImage.TYPE_INT_RGB);
		Graphics2D g2 = (Graphics2D) temp.getGraphics();
		g2.setColor(new Color(BIN0));
		g2.fillRect(0, 0, width + 2, height + 2);
		BufferedImage subimage = image.getSubimage(xb, yb, width, height);
		g2.drawImage(subimage, 1, 1, null);
		subimage = null;

		int[] table =
		// 0,1,2,3,4,5,6,7,8,9,0,1,2,3,4,5,6,7,8,9,0,1,2,3,4,5,6,7,8,9,0,1
		{ 0, 0, 0, 1, 0, 0, 1, 3, 0, 0, 3, 1, 1, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0,
				0, 2, 0, 2, 0, 3, 0, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 3, 0, 2, 2, 0,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0,
				0, 2, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 3, 0, 2, 0,
				0, 0, 3, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 1, 0, 0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
				0, 2, 3, 1, 3, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 0, 1, 0, 0, 0, 1, 0,
				0, 0, 0, 0, 0, 0, 0, 3, 3, 0, 1, 0, 0, 0, 0, 2, 2, 0, 0, 2, 0,
				0, 0 };

		int[] rgb = temp
				.getRGB(0, 0, width + 2, height + 2, null, 0, width + 2);
		int[] pixels = new int[rgb.length];
		int[] pixels2 = new int[rgb.length];
		for (int i = 0; i < rgb.length; i++) {
			if (rgb[i] == -1)
				pixels[i] = 255;
			else
				pixels[i] = 0;
		}
		System.arraycopy(pixels, 0, pixels2, 0, rgb.length);
		int pass = 0;
		int pixelsRemoved;

		do {
			g.setColor(Color.white);
			System.arraycopy(pixels, 0, pixels2, 0, rgb.length);
			pixelsRemoved = thin(pass++, table, width + 2, height + 2,
					width + 2, height + 2, pixels, pixels2);
			System.arraycopy(pixels, 0, pixels2, 0, rgb.length);
			pixelsRemoved = thin(pass++, table, width + 2, height + 2,
					width + 2, height + 2, pixels, pixels2);
		} while (pixelsRemoved > 0);

		// Assign the thinned image to img:
		// Fill array rgb with color values that correspond to the shades 0 and
		// 255
		for (int i = 0; i < pixels.length; i++) {
			int s = pixels[i];
			rgb[i] = new Color(s, s, s).getRGB();
		}

		// Form a new array from RGB; omit the one-pixel-border
		int[] borderlessRGB = new int[width * height];
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				borderlessRGB[y * width + x] = rgb[(y + 1) * (width + 2)
						+ (x + 1)];
			}
		}

		image.setRGB(xb, yb, width, height, borderlessRGB, 0, width);
		return image;
	}

	private int thin(int pass, int[] table, int width, int height,
			int roiWidth, int roiHeight, int[] pixels, int[] pixels2) {
		int xMin = 1, yMin = 1;
		int xMax = width - 1, yMax = height - 1;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
		int inc = roiHeight / 25;
		if (inc < 1)
			inc = 1;
		int bgColor = 255;
		int v, index, code;
		int offset, rowOffset = width;
		int pixelsRemoved = 0;
		for (int y = yMin; y < yMax; y++) {
			offset = xMin + y * width;
			for (int x = xMin; x < xMax; x++) {
				p5 = pixels2[offset];
				v = p5;

				if (v != bgColor) {
					p1 = pixels2[offset - rowOffset - 1];
					p2 = pixels2[offset - rowOffset];
					p3 = pixels2[offset - rowOffset + 1];
					p4 = pixels2[offset - 1];
					p6 = pixels2[offset + 1];
					p7 = pixels2[offset + rowOffset - 1];
					p8 = pixels2[offset + rowOffset];
					p9 = pixels2[offset + rowOffset + 1];
					index = 0;
					if (p1 != bgColor)
						index |= 1; // means "index = index | 1"
					if (p2 != bgColor)
						index |= 2;
					if (p3 != bgColor)
						index |= 4;
					if (p6 != bgColor)
						index |= 8;
					if (p9 != bgColor)
						index |= 16;
					if (p8 != bgColor)
						index |= 32;
					if (p7 != bgColor)
						index |= 64;
					if (p4 != bgColor)
						index |= 128;
					code = table[index];
					if ((pass & 1) == 1) { // odd pass
						if (code == 2 || code == 3) {
							v = bgColor;
							pixelsRemoved++;
						}
					} else { // even pass
						if (code == 1 || code == 3) {
							v = bgColor;
							pixelsRemoved++;
						}
					}
				}

				pixels[offset++] = v;
			}
		}
		return pixelsRemoved;
	}

	// Look for foreground constellations in which the middle pixel is in the
	// "center" of a right corner
	// Set those middle pixels on background shade BIN0
	public void deleteCornerPixels(BufferedImage image, Shape selection) {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int width = (int) bounds.getWidth();
		int height = (int) bounds.getHeight();
		int mi = BIN0, n = BIN0, ne = BIN0, nw = BIN0, s = BIN0, se = BIN0, sw = BIN0, e = BIN0, w = BIN0;

		try {
			for (int x = xb; x < width + xb; x++) {
				for (int y = yb; y < height + yb; y++) {
					mi = new Color(image.getRGB(x, y)).getRGB();

					// Check the pixel neighborship with an "inverse" logic:
					// A background pixel has the shade BIN0
					// A foreground pixel has either one of the shades BIN1 or
					// the border shade (gray)
					if (mi != BIN0) // Middle pixel is a foreground pixel (not a
					// background pixel)
					{
						n = new Color(image.getRGB(x, y - 1)).getRGB();
						nw = new Color(image.getRGB(x - 1, y - 1)).getRGB();
						ne = new Color(image.getRGB(x + 1, y - 1)).getRGB();
						w = new Color(image.getRGB(x - 1, y)).getRGB();
						e = new Color(image.getRGB(x + 1, y)).getRGB();
						s = new Color(image.getRGB(x, y + 1)).getRGB();
						sw = new Color(image.getRGB(x - 1, y + 1)).getRGB();
						se = new Color(image.getRGB(x + 1, y + 1)).getRGB();

						if (((s != BIN0) && (e != BIN0) && (nw != BIN1)
								&& (n != BIN1) && (ne != BIN1) && (w != BIN1) && (sw != BIN1))
								|| ((s != BIN0) && (w != BIN0) && (nw != BIN1)
										&& (n != BIN1) && (ne != BIN1)
										&& (e != BIN1) && (se != BIN1))
								|| ((n != BIN0) && (e != BIN0) && (nw != BIN1)
										&& (ne != BIN1) && (w != BIN1)
										&& (sw != BIN1) && (s != BIN1) && (se != BIN1))
								|| ((n != BIN0) && (w != BIN0) && (nw != BIN1)
										&& (ne != BIN1) && (e != BIN1)
										&& (sw != BIN1) && (s != BIN1) && (se != BIN1)))
							image.setRGB(x, y, new Color(BIN0).getRGB());
					}
				}
			}
		} catch (Exception ex) {
		}
		;
	}

	// Deletes single pixels at the periphery of thinned structures
	public void delete1PixelBranches(BufferedImage image, Shape selection) {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int width = (int) bounds.getWidth();
		int height = (int) bounds.getHeight();
		int c;
		int cnt = 0;
		Vector<Point> neighborsFG = new Vector<Point>();
		Vector<Point> pointsToDelete = new Vector<Point>();
		boolean pointRedundant = false;
		Point neighbor1, neighbor2;
		int pathCol = Color.green.getRGB();
		int size1 = 0, size2 = 0;
		int n1x, n1y, n2x, n2y;

		for (int x = xb; x < width + xb; x++) {
			for (int y = yb; y < height + yb; y++) {
				c = new Color(image.getRGB(x, y)).getRGB();

				if (c == BIN1) // The point is a foreground point
				{
					cnt = getN8FGCnt(image, x, y, BIN0);
					neighborsFG = getN8FGNeighbors(image, x, y);

					// Complex situation:
					// The point can be deleted either if this locally conserves
					// the connected foreground component,
					// or if the deletion results in a new component that falls
					// below a minimum size
					if (cnt == 6) {
						neighbor1 = neighborsFG.elementAt(0);
						n1x = neighbor1.x;
						n1y = neighbor1.y;
						neighbor2 = neighborsFG.elementAt(1);
						n2x = neighbor2.x;
						n2y = neighbor2.y;

						// The deletion locally conserves the connected
						// foreground component:
						// The two foreground neighbors of the pixel are
						// connected/direct neighbors (horiz./diag.)
						if (neighbor1.distance(neighbor2) < 1.42f) {
							pointsToDelete.add(new Point(x, y));
						} else // Not yet logically correct: The deletion
						// results in two separate components
						{

						}
					} else if (cnt == 7) // Simpler situation: The point can be
					// deleted under fixed conditions
					{
						pointRedundant = true;
						// Delete the pixel if each of its foreground neighbors
						// has at least
						// three foreground neighbors, including this pixel
						for (int i = 0; i < neighborsFG.size(); i++) {
							Point neighbor = neighborsFG.elementAt(i);
							if (getN8FGCnt(image, neighbor, BIN1) < 3) {
								pointRedundant = false;
								break;
							}
							;
						}
						if (pointRedundant)
							pointsToDelete.add(new Point(x, y));
					}
				}
			}
		}

		// Delete the points
		for (int i = 0; i < pointsToDelete.size(); i++) {
			Point p = pointsToDelete.elementAt(i);
			image.setRGB(p.x, p.y, BIN0);
		}
	}

	// Checks whether the selected portion of the image is two-leveled
	public boolean isBinarized(BufferedImage image, Shape selection) {
		// Checks whether the subimage specified by the selection is only made
		// of pixels of the shades 0 and 255 (pseudo-binarized)
		boolean binarized = true;
		int c = 0;

		if (selection == null)
			return false;

		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		try {
			for (int x = xb; x < w + xb; x++) {
				for (int y = yb; y < h + yb; y++) {
					c = new Color(image.getRGB(x, y)).getRed();
					if ((c != 0) && (c != 255))
						binarized = false;
				}
			}
		} catch (Exception ex) {
		}
		;

		return binarized;
	}

	public BufferedImage addBorderToImage(BufferedImage image)
			throws OutOfMemoryError {
		int w = image.getWidth();
		int h = image.getHeight();

		int borderWidth = 32;
		BufferedImage newImage = new BufferedImage(w + 2 * borderWidth, h + 2
				* borderWidth, BufferedImage.TYPE_INT_ARGB);

		int[] src = image.getRGB(0, 0, w, h, null, 0, w);
		newImage.setRGB(borderWidth, borderWidth, w, h, src, 0, w);

		return newImage;
	}

	public BufferedImage removeBorderFromImage(BufferedImage image)
			throws OutOfMemoryError {
		int width = image.getWidth();
		int height = image.getHeight();
		int borderWidth = 32;
		int newWidth = width - 2 * borderWidth;
		int newHeight = height - 2 * borderWidth;

		BufferedImage newImage = new BufferedImage(width - 2 * borderWidth,
				height - 2 * borderWidth, BufferedImage.TYPE_INT_ARGB);
		int[] src = image.getRGB(borderWidth, borderWidth, newWidth, newHeight,
				null, 0, newWidth);
		newImage.setRGB(0, 0, newWidth, newHeight, src, 0, newWidth);

		return newImage;
	}

	public HashSet<Integer> collectSelectedColors(BufferedImage image,
			Shape selection) throws OutOfMemoryError {
		HashSet<Integer> colorSet = new HashSet<Integer>();

		if ((selection != null) && (image != null)) {
			Rectangle2D boundingBox = selection.getBounds2D();
			int x = (int) boundingBox.getX();
			int y = (int) boundingBox.getY();
			int w = (int) boundingBox.getWidth();
			int h = (int) boundingBox.getHeight();

			for (int i = x; i < x + w; i++) {
				for (int j = y; j < y + h; j++) {
					colorSet.add(image.getRGB(i, j));
				}
			}
		}

		return colorSet;
	}

	// Return the subimage under the selection area
	public BufferedImage getSubImage(BufferedImage image, Shape selection,
			boolean drawBorder) throws OutOfMemoryError {
		// Return a subimage that matches the bounding box around the selection
		// If the resulting image is too small, return null
		// Note that the subimage used and the original image share the same
		// data buffer (may lead to confusion!)
		if ((selection != null) && (image != null)) {
			Rectangle2D boundingBox = selection.getBounds2D();
			int x = (int) boundingBox.getX();
			int y = (int) boundingBox.getY();
			int width = (int) boundingBox.getWidth();
			int height = (int) boundingBox.getHeight();
			if ((width < 10) && (height < 10))
				return null;

			BufferedImage newImage = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_ARGB);
			if (drawBorder)
				newImage = addBorderToImage(image.getSubimage(x, y, width,
						height));
			else
				newImage = image.getSubimage(x, y, width, height);
			return newImage;
		}
		return null;
	}

	public BufferedImage copySubImage(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		if ((selection != null) && (image != null)) {
			Rectangle2D boundingBox = selection.getBounds2D();
			int x = (int) boundingBox.getX();
			int y = (int) boundingBox.getY();
			int width = (int) boundingBox.getWidth();
			int height = (int) boundingBox.getHeight();
			if ((width < 30) && (height < 30))
				return null;

			BufferedImage newImage = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_ARGB);
			int[] rgb = image.getRGB(x, y, width, height, null, 0, width);
			newImage.setRGB(0, 0, width, height, rgb, 0, width);
			rgb = null;
			return newImage;
		}
		return null;
	}

	// Return an image which is plain white except for the selection area
	public BufferedImage trimImage(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		Area imageArea = new Area(new Rectangle2D.Double(0, 0,
				image.getWidth(), image.getHeight()));
		Area trimArea = new Area(selection.getBounds2D());
		imageArea.subtract(trimArea);
		Graphics2D g2 = (Graphics2D) image.getGraphics();
		g2.fill(imageArea);

		return addBorderToImage(image);
	}

	public BufferedImage scaleImage2x(BufferedImage image, boolean drawBorder)
			throws OutOfMemoryError {
		int width = image.getWidth();
		int height = image.getHeight();
		int borderWidth = 32;

		// Get the area of the source image without the border
		// -1 added, otherwise complete images are not skeletonized after 2x
		// scaling (why?)
		BufferedImage borderlessImage = image.getSubimage(borderWidth,
				borderWidth, width - borderWidth - 1, height - borderWidth - 1);

		// Build a new empty image
		BufferedImage newImage = new BufferedImage(2 * (width - borderWidth),
				2 * (height - borderWidth), BufferedImage.TYPE_INT_RGB);

		// Scale the image
		Graphics g = newImage.getGraphics();
		g.drawImage(borderlessImage, 0, 0, 2 * width, 2 * height, null);
		g.dispose();

		// Draw a border around the image if option is true
		if (drawBorder)
			newImage = addBorderToImage(newImage);

		return newImage;
	}

	public BufferedImage scaleImage05x(BufferedImage image, boolean drawBorder)
			throws OutOfMemoryError {
		int width = image.getWidth();
		int height = image.getHeight();
		int borderWidth = 32;

		// Get the area of the source image without the border
		BufferedImage borderlessImage = image.getSubimage(borderWidth,
				borderWidth, width - borderWidth, height - borderWidth);

		// Build a new empty image
		BufferedImage newImage = new BufferedImage(
				(int) (0.5 * (width - borderWidth)),
				(int) (0.5 * (height - borderWidth)),
				BufferedImage.TYPE_INT_RGB);

		// Scale the image
		Graphics g = newImage.getGraphics();
		g.drawImage(borderlessImage, 0, 0, (int) (0.5 * width),
				(int) (0.5 * height), null);
		g.dispose();

		// Draw a border around the image if option is true
		if (drawBorder)
			newImage = addBorderToImage(newImage);

		return newImage;
	}

	public int[] computeGrayscaleHistogram(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		// Prerequisite: BufferedImage of type TYPE_INT_RGB, not
		// alphaPremultiplied, component size 8
		if (image != null) {
			histogram = new int[256];

			// Return a subimage that matches the bounding box around the
			// selection
			Rectangle2D boundingBox = selection.getBounds2D();
			int x = (int) boundingBox.getX();
			int y = (int) boundingBox.getY();
			int w = (int) boundingBox.getWidth();
			int h = (int) boundingBox.getHeight();

			for (int i = x; i < x + w; i++) {
				for (int j = y; j < y + h; j++) {
					int c = new Color(image.getRGB(i, j)).getRed();
					histogram[c]++;
				}
			}
			return histogram;
		}
		return null;
	}

	public BufferedImage regionLabeling(BufferedImage image, Shape selection)
			throws OutOfMemoryError {
		// Return a subimage that matches the bounding box around the selection
		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX() + 1;
		int y = (int) boundingBox.getY() + 1;
		int w = (int) boundingBox.getWidth() - 1;
		int h = (int) boundingBox.getHeight() - 1;
		int regionColor = new Color(0, 0, 1).getRGB();

		for (int i = x; i < x + w; i++) {
			for (int j = y; j < y + h; j++) {
				int seedCol = image.getRGB(i, j);
				if (seedCol == BIN1) {
					localFloodFill8NB(image, selection, i, j, BIN1,
							(regionColor++) * 31);
					int a = (int) Math.min(255, (int) 255
							* (Math.random() + 0.1));
					int b = (int) Math.min(255, (int) 255
							* (Math.random() + 0.1));
					int c = (int) Math.min(255, (int) 255
							* (Math.random() + 0.1));
					regionColor = new Color(a, b, c).getRGB();
				}
			}
		}

		return image;
	}

	public int localFloodFill8NB(BufferedImage image, Shape selection, int i,
			int j, int fromCol, int toCol) {
		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX();
		int y = (int) boundingBox.getY();
		int w = (int) boundingBox.getWidth() - 1;
		int h = (int) boundingBox.getHeight() - 1;
		int filledPixels = 0;

		Stack<Point> stack = new Stack<Point>();
		stack.push(new Point(i, j));

		while (stack.isEmpty() == false) {
			Point p = stack.pop();
			int x0 = p.x;
			int y0 = p.y;
			int pixelCol = image.getRGB(x0, y0);

			try {
				if ((pixelCol == fromCol) && (x0 >= x) && (x0 <= x + w)
						&& (y0 >= y) && (y0 <= y + h)) {
					image.setRGB(x0, y0, toCol);
					filledPixels++;

					stack.push(new Point(x0 + 1, y0));
					stack.push(new Point(x0, y0 + 1));
					stack.push(new Point(x0, y0 - 1));
					stack.push(new Point(x0 - 1, y0));

					stack.push(new Point(x0 - 1, y0 - 1));
					stack.push(new Point(x0 + 1, y0 - 1));
					stack.push(new Point(x0 - 1, y0 + 1));
					stack.push(new Point(x0 + 1, y0 + 1));
				}
			} catch (OutOfMemoryError e) {
				stack.clear();
				System.gc();
			}
		}
		stack.clear();

		return filledPixels; // Number of pixels colored in blue
	}

	public int localFloodFill8NBLimit(BufferedImage image, Shape selection,
			int i, int j, int fromCol, int toCol, int limit) {
		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX();
		int y = (int) boundingBox.getY();
		int w = (int) boundingBox.getWidth() - 1;
		int h = (int) boundingBox.getHeight() - 1;
		Vector<Point> temp = new Vector<Point>();
		Stack<Point> stack = new Stack<Point>();

		stack.push(new Point(i, j));
		temp.add(new Point(i, j));

		while (!stack.empty()) {
			Point p = stack.pop();
			if (temp.size() == limit)
				break;
			int x0 = p.x;
			int y0 = p.y;

			int pixelCol = image.getRGB(x0, y0);

			try {
				if ((pixelCol == fromCol) && (x0 >= x) && (x0 <= x + w)
						&& (y0 >= y) && (y0 <= y + h)) {
					image.setRGB(x0, y0, toCol);
					temp.add(new Point(x0, y0));

					stack.push(new Point(x0 + 1, y0));
					stack.push(new Point(x0, y0 + 1));
					stack.push(new Point(x0, y0 - 1));
					stack.push(new Point(x0 - 1, y0));

					stack.push(new Point(x0 - 1, y0 - 1));
					stack.push(new Point(x0 + 1, y0 - 1));
					stack.push(new Point(x0 - 1, y0 + 1));
					stack.push(new Point(x0 + 1, y0 + 1));
				}
			} catch (OutOfMemoryError e) {
				stack.clear();
				System.gc();
			}
		}

		// Restore the pixels
		for (int k = 0; k < temp.size(); k++) {
			Point p = temp.elementAt(k);
			image.setRGB(p.x, p.y, BIN1);
		}

		return temp.size();
	}

	// Performs an Octree color quantization on the image
	// reducing the color set to {numColors} different colors
	public BufferedImage quantizeColors(BufferedImage image, int numColors)
			throws OutOfMemoryError {
		BufferedImage reducedImage;

		// Cut off the image border
		BufferedImage borderlessImage = removeBorderFromImage(image);
		// Same effect: BufferedImage borderlessImage = getSubImage(image,
		// selection, false);

		// Get the image data as an RGB data array
		int[][] rgbData = (int[][]) imageBuffer
				.returnImageAs2DArray(borderlessImage);
		// Quantize the image in-place into an indexed image, return the index
		// palette
		int[] reducedPalette = Quantize.quantizeImage(rgbData, numColors);
		// Fill an array with RGB values using the palette as dictionary
		imageBuffer.makeRGBFromIndexTable(rgbData, reducedPalette);
		// Build an image that contains the RGB array as raster
		reducedImage = imageBuffer.return2DArrayAsImage(rgbData);

		return (BufferedImage) addBorderToImage(reducedImage);
	}

	// Globally replaces a set of colors with a new color
	public synchronized BufferedImage replaceColorSet(BufferedImage image,
			Shape selection, HashSet<Integer> fromColors, int fillCol)
			throws OutOfMemoryError {
		if (fromColors.size() == 0)
			return image; // Return the unmodified image if no colors are in the
		// set

		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX();
		int y = (int) boundingBox.getY();
		int w = (int) boundingBox.getWidth() - 1;
		int h = (int) boundingBox.getHeight() - 1;
		int c, ic;

		for (int i = x; i < x + w; i++) {
			for (int j = y; j < y + h; j++) {
				ic = image.getRGB(i, j);
				for (Iterator<Integer> it = fromColors.iterator(); it.hasNext();) {
					c = it.next();
					if (ic == c)
						image.setRGB(i, j, fillCol);
				}
			}
		}

		return image;

	}

	public synchronized BufferedImage keepColorSet(BufferedImage image,
			Shape selection, HashSet<Integer> fromColors, int fillCol)
			throws OutOfMemoryError {
		if (fromColors.size() == 0)
			return image; // Return the unmodified image if no colors are in the
		// set
		if (selection == null)
			return image;

		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX();
		int y = (int) boundingBox.getY();
		int w = (int) boundingBox.getWidth() - 1;
		int h = (int) boundingBox.getHeight() - 1;
		int c, ic;
		boolean notInSet;

		for (int i = x; i < x + w; i++) {
			for (int j = y; j < y + h; j++) {
				ic = image.getRGB(i, j);
				notInSet = true;
				for (Iterator<Integer> it = fromColors.iterator(); it.hasNext();) {
					c = it.next();
					if (ic == c)
						notInSet = false;
				}
				if (notInSet)
					image.setRGB(i, j, fillCol);
			}
		}

		return image;
	}

	public synchronized void localFloodFill4NB(BufferedImage image,
			Shape selection, int i, int j, int fromCol, int toCol)
			throws OutOfMemoryError {
		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX();
		int y = (int) boundingBox.getY();
		int w = (int) boundingBox.getWidth() - 1;
		int h = (int) boundingBox.getHeight() - 1;
		Point p;

		if (image.getRGB(i, j) == toCol)
			return;

		Stack<Point> stack = new Stack<Point>();
		stack.push(new Point(i, j));

		while (stack.isEmpty() == false) {
			p = stack.pop();
			int x0 = p.x;
			int y0 = p.y;
			int pixelCol = image.getRGB(x0, y0);

			try {
				// TODO: Ignores the bounding box
				if ((x0 >= x) && (x0 <= x + w) && (y0 >= y) && (y0 <= y + h)) {
					if (pixelCol == fromCol) {
						image.setRGB(x0, y0, toCol);
						stack.push(new Point(x0 + 1, y0));
						stack.push(new Point(x0, y0 + 1));
						stack.push(new Point(x0, y0 - 1));
						stack.push(new Point(x0 - 1, y0));
					}
				}
			} catch (OutOfMemoryError e) {
				stack.clear();
			}
		}
		stack.clear();
	}

	public boolean getN8containsColor(BufferedImage image, Point l, int c) {
		int x = l.x;
		int y = l.y;

		if ((image.getRGB(x - 1, y - 1) == c) || (image.getRGB(x, y - 1) == c)
				|| (image.getRGB(x + 1, y - 1) == c)
				|| (image.getRGB(x - 1, y) == c)
				|| (image.getRGB(x + 1, y) == c)
				|| (image.getRGB(x - 1, y + 1) == c)
				|| (image.getRGB(x, y + 1) == c)
				|| (image.getRGB(x + 1, y + 1) == c))
			return true;

		return false;
	}

	public Point getN8PointWithColor(BufferedImage image, Point l,
			int whichPointPreferred, int c) {
		int x = l.x;
		int y = l.y;

		if ((whichPointPreferred == 1) && (image.getRGB(x - 1, y - 1) == c))
			return new Point(x - 1, y - 1);
		else if ((whichPointPreferred == 2) && (image.getRGB(x, y - 1) == c))
			return new Point(x, y - 1);
		else if ((whichPointPreferred == 3)
				&& (image.getRGB(x + 1, y - 1) == c))
			return new Point(x + 1, y - 1);
		else if ((whichPointPreferred == 4) && (image.getRGB(x - 1, y) == c))
			return new Point(x - 1, y);
		else if ((whichPointPreferred == 5) && (image.getRGB(x, y) == c))
			return new Point(x, y);
		else if ((whichPointPreferred == 6) && (image.getRGB(x + 1, y) == c))
			return new Point(x + 1, y);
		else if ((whichPointPreferred == 7)
				&& (image.getRGB(x - 1, y + 1) == c))
			return new Point(x - 1, y + 1);
		else if ((whichPointPreferred == 8) && (image.getRGB(x, y + 1) == c))
			return new Point(x, y + 1);
		else if ((whichPointPreferred == 9)
				&& (image.getRGB(x + 1, y + 1) == c))
			return new Point(x + 1, y + 1);

		if (image.getRGB(x - 1, y - 1) == c)
			return new Point(x - 1, y - 1);
		if (image.getRGB(x, y - 1) == c)
			return new Point(x, y - 1);
		if (image.getRGB(x + 1, y - 1) == c)
			return new Point(x + 1, y - 1);
		if (image.getRGB(x - 1, y) == c)
			return new Point(x - 1, y);
		if (image.getRGB(x, y) == c)
			return new Point(x, y);
		if (image.getRGB(x + 1, y) == c)
			return new Point(x + 1, y);
		if (image.getRGB(x - 1, y + 1) == c)
			return new Point(x - 1, y + 1);
		if (image.getRGB(x, y + 1) == c)
			return new Point(x, y + 1);
		if (image.getRGB(x + 1, y + 1) == c)
			return new Point(x + 1, y + 1);
		return null;
	}

	// Returns the point with selected color first found in the neighborship
	public Point getN8PointWithColor(BufferedImage image, Point l, int c) {
		int x = l.x;
		int y = l.y;

		if (image.getRGB(x - 1, y - 1) == c)
			return new Point(x - 1, y - 1);
		if (image.getRGB(x, y - 1) == c)
			return new Point(x, y - 1);
		if (image.getRGB(x + 1, y - 1) == c)
			return new Point(x + 1, y - 1);
		if (image.getRGB(x - 1, y) == c)
			return new Point(x - 1, y);
		if (image.getRGB(x, y) == c)
			return new Point(x, y);
		if (image.getRGB(x + 1, y) == c)
			return new Point(x + 1, y);
		if (image.getRGB(x - 1, y + 1) == c)
			return new Point(x - 1, y + 1);
		if (image.getRGB(x, y + 1) == c)
			return new Point(x, y + 1);
		if (image.getRGB(x + 1, y + 1) == c)
			return new Point(x + 1, y + 1);

		return null;
	}

	public void color8Neighborship(BufferedImage image, Point l, int bgColor) {
		Graphics2D g2 = (Graphics2D) image.getGraphics();
		int x = l.x;
		int y = l.y;
		int middleShade = image.getRGB(x, y);
		g2.setColor(new Color(bgColor));
		g2.fillRect(x - 1, y - 1, 3, 3);
		g2.setColor(new Color(middleShade));
		g2.drawLine(x, y, 1, 1);
	}

	// Returns the number of foreground pixels around point l
	public int getN8FGCnt(BufferedImage image, Point l, int fgColor) {
		int x = l.x;
		int y = l.y;
		int cnt = 0;

		if (image.getRGB(x - 1, y - 1) == fgColor)
			cnt++;
		if (image.getRGB(x, y - 1) == fgColor)
			cnt++;
		if (image.getRGB(x + 1, y - 1) == fgColor)
			cnt++;
		if (image.getRGB(x - 1, y) == fgColor)
			cnt++;
		if (image.getRGB(x + 1, y) == fgColor)
			cnt++;
		if (image.getRGB(x - 1, y + 1) == fgColor)
			cnt++;
		if (image.getRGB(x, y + 1) == fgColor)
			cnt++;
		if (image.getRGB(x + 1, y + 1) == fgColor)
			cnt++;

		return cnt;
	}

	public int getN8FGCnt(BufferedImage image, int x, int y, int c) {
		return getN8FGCnt(image, new Point(x, y), c);
	}

	// Returns the number of pixels around point l that have not background
	// color bgColor
	public int getN8NonBGCnt(BufferedImage image, Point l, int bgColor) {
		int x = l.x;
		int y = l.y;
		int cnt = 0;

		if (image.getRGB(x - 1, y - 1) != bgColor)
			cnt++;
		if (image.getRGB(x, y - 1) != bgColor)
			cnt++;
		if (image.getRGB(x + 1, y - 1) != bgColor)
			cnt++;
		if (image.getRGB(x - 1, y) != bgColor)
			cnt++;
		if (image.getRGB(x + 1, y) != bgColor)
			cnt++;
		if (image.getRGB(x - 1, y + 1) != bgColor)
			cnt++;
		if (image.getRGB(x, y + 1) != bgColor)
			cnt++;
		if (image.getRGB(x + 1, y + 1) != bgColor)
			cnt++;

		return cnt;
	}

	public Vector<Point> getN8FGNeighbors(BufferedImage image, int x, int y) {
		Vector<Point> fgNeighbors = new Vector<Point>();

		if (image.getRGB(x - 1, y - 1) == BIN1)
			fgNeighbors.add(new Point(x - 1, y - 1));
		if (image.getRGB(x, y - 1) == BIN1)
			fgNeighbors.add(new Point(x, y - 1));
		if (image.getRGB(x + 1, y - 1) == BIN1)
			fgNeighbors.add(new Point(x + 1, y - 1));
		if (image.getRGB(x - 1, y) == BIN1)
			fgNeighbors.add(new Point(x - 1, y));
		if (image.getRGB(x + 1, y) == BIN1)
			fgNeighbors.add(new Point(x + 1, y));
		if (image.getRGB(x - 1, y + 1) == BIN1)
			fgNeighbors.add(new Point(x - 1, y + 1));
		if (image.getRGB(x, y + 1) == BIN1)
			fgNeighbors.add(new Point(x, y + 1));
		if (image.getRGB(x + 1, y + 1) == BIN1)
			fgNeighbors.add(new Point(x + 1, y + 1));

		return fgNeighbors;
	}

	public Vector<Point> getN8NonBGNeighbors(BufferedImage image, int x, int y) {
		Vector<Point> fgNeighbors = new Vector<Point>();

		if (image.getRGB(x - 1, y - 1) != BIN0)
			fgNeighbors.add(new Point(x - 1, y - 1));
		if (image.getRGB(x, y - 1) != BIN0)
			fgNeighbors.add(new Point(x, y - 1));
		if (image.getRGB(x + 1, y - 1) != BIN0)
			fgNeighbors.add(new Point(x + 1, y - 1));
		if (image.getRGB(x - 1, y) != BIN0)
			fgNeighbors.add(new Point(x - 1, y));
		if (image.getRGB(x + 1, y) != BIN0)
			fgNeighbors.add(new Point(x + 1, y));
		if (image.getRGB(x - 1, y + 1) != BIN0)
			fgNeighbors.add(new Point(x - 1, y + 1));
		if (image.getRGB(x, y + 1) != BIN0)
			fgNeighbors.add(new Point(x, y + 1));
		if (image.getRGB(x + 1, y + 1) != BIN0)
			fgNeighbors.add(new Point(x + 1, y + 1));

		return fgNeighbors;
	}

	// Return all points in color c within the specified area that have n direct
	// foreground neighbors
	public Vector<Point> getNNeighborPixels(BufferedImage image, int n,
			Shape selection, int c) {
		if (image == null)
			return null;
		Vector<Point> v = new Vector<Point>();
		Point p;
		// Return a subimage that matches the bounding box around the selection
		Rectangle2D boundingBox = selection.getBounds2D();
		int x = (int) boundingBox.getX();
		int y = (int) boundingBox.getY();
		int w = (int) boundingBox.getWidth();
		int h = (int) boundingBox.getHeight();

		for (int i = x; i < x + w; i++) {
			for (int j = y; j < y + h; j++) {
				p = new Point(i, j);
				if (image.getRGB(i, j) == c) {
					if (getN8FGCnt(image, i, j, c) == n) {
						v.add(p);
					}
				}
			}
		}
		return v;
	}

	// Return all pixels within the 8-neighborship of the specified point that
	// have a color other than those in colorsToExclude:
	// background Color, border color, wizard.pathColor,
	public Vector<Pixel> getN8_ExcludeColors(BufferedImage image,
			boolean extended, Point l, Color[] excludeColors) {

		Vector<Pixel> pixels = new Vector<Pixel>();
		boolean allowedColor;
		int color;
		int xm = l.x;
		int ym = l.y;
		for (int x = xm - 1; x <= xm + 1; x++) {
			for (int y = ym - 1; y <= ym + 1; y++) {
				color = image.getRGB(x, y);
				allowedColor = true;

				// Excluded color is disallowed
				for (int i = 0; i < excludeColors.length; i++) {
					int excludeCol = excludeColors[i].getRGB();
					if (color == excludeCol)
						allowedColor = false;
				}

				if (extended) {
					// One of the signal colors or the foreground color are
					// allowed
					// if ((!UniqueColorSource.isSignalColor(color)) && (color
					// != BIN1)) allowedColor = false;
					if (color != BIN1)
						allowedColor = false;
				}

				if (allowedColor)
					pixels.add(new Pixel(x, y, color));
			}
		}
		return pixels;
	}

	// Returns the nearest point with the color specified within a square with
	// center l
	// Returns either a Point object or null
	public Point getNearestPointWithColor(BufferedImage image, Point l,
			int rectWidth, int wantedCol) {
		Point nearestPoint = null;
		double distance = 0, minDistance = Integer.MAX_VALUE;
		int halfRectWidth = (int) (0.5 * rectWidth);
		int x0 = Math.max(0, l.x - halfRectWidth);
		int x1 = Math.min(image.getWidth(), l.x + halfRectWidth);
		int y0 = Math.max(0, l.y - halfRectWidth);
		int y1 = Math.min(image.getHeight(), l.y + halfRectWidth);

		for (int x = x0; x <= x1; x++) {
			for (int y = y0; y <= y1; y++) {
				Point p = new Point(x, y);
				int color = image.getRGB(x, y);

				if (color == wantedCol) {
					distance = l.distance(p);
					if (distance < minDistance) {
						minDistance = distance;
						nearestPoint = p;
					}
				}
			}
		}
		return nearestPoint;
	}

	public void colorPixels(Vector<Point> v) {
		BufferedImage image = imageBuffer.getProcessedImage();
		Graphics2D g = (Graphics2D) (image.getGraphics());
		g.setColor(Color.red);

		for (int i = 0; i < v.size(); i++) {
			Point p = v.elementAt(i);
			int x = p.x;
			int y = p.y;
			image.setRGB(x, y, Color.green.getRGB());
		}
	}

	// Sets all pixels within the 8-neighborship of point l to black whose color
	// is
	// not the background shade (normally BIN0)
	// If the argument fg is BIN0, bg is BIN1 and vice versa
	public void blackenN8FGPixels(Point l, int fg) {
		BufferedImage image = imageBuffer.getProcessedImage();
		if (l != null) {
			int x = l.x;
			int y = l.y;
			int bg = BIN0;
			if (fg == BIN1)
				bg = BIN0;
			else {
				fg = BIN0;
				bg = BIN1;
			}
			;

			if (image.getRGB(x - 1, y - 1) != BIN0)
				image.setRGB(x - 1, y - 1, BIN1);
			if (image.getRGB(x, y - 1) != BIN0)
				image.setRGB(x, y - 1, BIN1);
			if (image.getRGB(x + 1, y - 1) != BIN0)
				image.setRGB(x + 1, y - 1, BIN1);
			if (image.getRGB(x - 1, y) != BIN0)
				image.setRGB(x - 1, y, BIN1);
			if (image.getRGB(x + 1, y) != BIN0)
				image.setRGB(x + 1, y, BIN1);
			if (image.getRGB(x - 1, y + 1) != BIN0)
				image.setRGB(x - 1, y + 1, BIN1);
			if (image.getRGB(x, y + 1) != BIN0)
				image.setRGB(x, y + 1, BIN1);
			if (image.getRGB(x + 1, y + 1) != BIN0)
				image.setRGB(x + 1, y + 1, BIN1);
		}
	}

	// Sets all pixels within the image to black whose color is
	// not the background shade (normally BIN0)
	// If the argument fg is BIN0, bg is BIN1 and vice versa
	public void blackenFGPixels(BufferedImage image, int bg, Shape selection) {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		int fg;
		if (bg == BIN0)
			fg = BIN1;
		else {
			fg = BIN0;
			bg = BIN1;
		}
		;

		for (int x = xb; x < xb + w; x++) {
			for (int y = yb; y < yb + h; y++) {
				if (image.getRGB(x, y) != bg)
					image.setRGB(x, y, fg);
			}
		}

	}

	// Color all node locations with the specified shade
	public void colorPixels(Vector<Point> nodes, int shade) {
		BufferedImage image = imageBuffer.getProcessedImage();

		for (int i = 0; i < nodes.size(); i++) {
			Point p = nodes.elementAt(i);
			int x = p.x;
			int y = p.y;
			image.setRGB(x, y, shade);
		}
	}

	// Color all node locations with the specified shade
	public void colorNodePositions(Vector<TreeNode> nodes, Color shade) {
		BufferedImage image = imageBuffer.getProcessedImage();
		int c = shade.getRGB();

		for (int i = 0; i < nodes.size(); i++) {
			Point p = (Point) nodes.elementAt(i).getLocation();
			int x = p.x;
			int y = p.y;
			image.setRGB(x, y, c);
		}
	}

	public Shape getShape(Point upperLeft, Point lowerRight) {
		int x = 0, y = 0;
		int x1 = upperLeft.x;
		int y1 = upperLeft.y;
		int x2 = lowerRight.x;
		int y2 = lowerRight.y;
		int w = Math.abs(x2 - x1);
		int h = Math.abs(y2 - y1);
		if (x2 > x1)
			x = x1;
		else
			x = x2;
		if (y2 > y1)
			y = y1;
		else
			y = y2;
		return new Rectangle2D.Float(x, y, w, h);
	}

	// Set all pixels to foreground shade (black) that have not the background
	// shade (white)
	public BufferedImage swapPixelColors(BufferedImage image, Shape selection,
			int fromColor, int toColor) {
		Rectangle2D bounds = selection.getBounds2D();
		int xb = (int) bounds.getX();
		int yb = (int) bounds.getY();
		int w = (int) bounds.getWidth();
		int h = (int) bounds.getHeight();

		for (int i = xb; i < xb + w; i++) {
			for (int j = yb; j < yb + h; j++) {
				if (image.getRGB(i, j) == fromColor)
					image.setRGB(i, j, toColor);
			}
		}

		return image;
	}

	public void buildDistancesSimple(Vector<TreeNode> nodes,
			HashMap<Point, HashMap<Point, Integer>> distances) {
		Stack<Point> pred = new Stack<Point>(); // Holds the pixels currently
		// visited
		Stack<Point> succ = new Stack<Point>(); // Hold the pixels from their
		// Moore-neighborship
		BufferedImage image = imageBuffer.getProcessedImage();
		int red = Color.red.getRGB();
		int blue = Color.blue.getRGB();
		int d = 0;
		Point current = null;
		Vector<Point> temp = new Vector<Point>();

		for (int i = 0; i < nodes.size(); i++) {
			d = 0; // Reset distance to seed position
			TreeNode node = nodes.elementAt(i);
			Point seed = node.getLocation(); // pos is the location of the node

			// For a new node at location {pos}, create and fill its distance
			// hash
			HashMap<Point, Integer> dist = new HashMap<Point, Integer>();
			dist.put(seed, 0); // Store the seed position with distance 0
			pred.push(seed); // Make it the first position to visit
			image.setRGB(seed.x, seed.y, blue);

			do {
				d++;
				succ.clear();

				while (!pred.empty()) {
					current = pred.pop();
					int x = current.x;
					int y = current.y;
					temp.add(new Point(x, y));

					if (getN8containsColor(image, current, red)) {
						if (image.getRGB(x - 1, y - 1) == red)
							current = new Point(x - 1, y - 1);
						else if (image.getRGB(x, y - 1) == red)
							current = new Point(x, y - 1);
						else if (image.getRGB(x + 1, y - 1) == red)
							current = new Point(x + 1, y - 1);
						else if (image.getRGB(x - 1, y) == red)
							current = new Point(x - 1, y);
						else if (image.getRGB(x + 1, y) == red)
							current = new Point(x + 1, y);
						else if (image.getRGB(x - 1, y + 1) == red)
							current = new Point(x - 1, y + 1);
						else if (image.getRGB(x, y + 1) == red)
							current = new Point(x, y + 1);
						else if (image.getRGB(x + 1, y + 1) == red)
							current = new Point(x + 1, y + 1);

						// Store the final point in the path
						if (current != null) {
							int a = current.x;
							int b = current.y;
							dist.put(current, d);

							// No need to investigate the Moore-neighborship
							pred.remove(new Point(a - 1, b - 1));
							pred.remove(new Point(a, b - 1));
							pred.remove(new Point(a + 1, b - 1));
							pred.remove(new Point(a - 1, b));
							pred.remove(new Point(a + 1, b));
							pred.remove(new Point(a - 1, b + 1));
							pred.remove(new Point(a, b + 1));
							pred.remove(new Point(a + 1, b + 1));
							blackenN8FGPixels(current, BIN1);

						}
					} else
						investigateMooreNeighborship(image, current, succ,
								dist, d);
				}
				pred.addAll(succ); // The successors become the new predecessors
			} while (!succ.empty());

			// Save all distance calculations for the seed point (the node
			// position)
			distances.put(seed, dist);

			// Restore colors, empty stack
			pred.empty();
			succ.empty();
			for (int j = 0; j < temp.size(); j++) {
				Point p = temp.elementAt(j);
				image.setRGB(p.x, p.y, BIN1);
			}
			;
			temp.clear();
			image.setRGB(seed.x, seed.y, red);
		}
	} // end of buildDistancesSimple

	public void buildDistancesElaborate(Vector<TreeNode> nodes,
			HashMap<Point, HashMap<Point, Integer>> distances) {
		Stack<Point> pred = new Stack<Point>(); // Holds the pixels currently
		// visited
		Stack<Point> succ = new Stack<Point>(); // Hold the pixels from their
		// Moore-neighborship
		BufferedImage image = imageBuffer.getProcessedImage();
		int red = Color.red.getRGB();
		int blue = Color.blue.getRGB();
		int d = 0;
		Point current = null;
		Vector<Point> temp = new Vector<Point>();

		for (int i = 0; i < nodes.size(); i++) {
			d = 0; // Reset distance to seed position
			TreeNode node = nodes.elementAt(i);
			Point seed = node.getLocation(); // pos is the location of the node

			// For a new node at location {pos}, create and fill its distance
			// hash
			HashMap<Point, Integer> dist = new HashMap<Point, Integer>();
			dist.put(seed, 0); // Store the seed position with distance 0
			pred.push(seed); // Make it the first position to visit
			image.setRGB(seed.x, seed.y, blue);

			do {
				d++;
				succ.clear();

				while (!pred.empty()) {
					current = pred.pop();

					temp.add(current);
					Point redPoint = null;

					// Check whether there is a red point in vicinity
					redPoint = getNearestPointWithColor(image, current,
							wizard.lookaheadDistance, red);
					if ((redPoint != null))
						elongatePath(image, current, redPoint, d, dist, temp,
								pred);
					else
						investigateMooreNeighborship(image, current, succ,
								dist, d);
				}
				pred.addAll(succ); // The successors become the new predecessors
			} while (!succ.empty());

			Set<Point> distSet = dist.keySet();
			Iterator<Point> it = distSet.iterator();

			// Save all distance calculations for the seed point (the node
			// position)
			distances.put(seed, dist);

			// Restore colors, empty stack
			pred.empty();
			succ.empty();
			for (int j = 0; j < temp.size(); j++) {
				Point p = temp.elementAt(j);
				image.setRGB(p.x, p.y, BIN1);
			}
			;
			temp.clear();

			image.setRGB(seed.x, seed.y, red);
		}
	}

	private boolean elongatePath(BufferedImage image, Point from,
			Point redPoint, int distanceFromSeed,
			HashMap<Point, Integer> absDistances, Vector<Point> triedPoints,
			Vector<Point> predecessors) {
		// Try to find one of the shortest foreground paths between points from
		// and to
		// If a foreground distance is found, add the path segment to the path,
		// do nothing otherwise
		// It is sufficient to look at a rectangle with the diameter of the
		// Manhattan distance.
		int relD = 0;
		int minX = Math.max(0, Math.min(from.x, redPoint.x));
		int maxX = Math.min(image.getWidth(), Math.max(from.x, redPoint.x));
		int minY = Math.max(0, Math.min(from.y, redPoint.y));
		int maxY = Math.min(image.getHeight(), Math.max(from.y, redPoint.y));
		Stack<Point> pred = new Stack<Point>(); // Holds the pixels currently
		// visited
		Stack<Point> succ = new Stack<Point>(); // Hold the pixels from their
		// Moore-neighborship
		HashMap<Point, Integer> relDistances = new HashMap<Point, Integer>();
		relDistances.put(from, 0); // Store the seed position with distance 0
		pred.push(from); // Make it the first position to visit
		int red = Color.red.getRGB();
		int blue = Color.blue.getRGB();
		image.setRGB(redPoint.x, redPoint.y, red);
		image.setRGB(from.x, from.y, blue);

		do {
			relD++;
			succ.clear();

			while (!pred.empty()) {
				Point p = pred.pop();
				triedPoints.add(p);
				int x = p.x;
				int y = p.y;

				// Investigate the Moore neighborship within the allowed window
				// defined by the two points
				if ((x >= minX) && (x <= maxX) && (y >= minY) && (y <= maxY)) {
					Point q = getN8PointWithColor(image, p, red);
					if (q == null) {

						if (image.getRGB(x - 1, y - 1) == BIN1) {
							succ.push(new Point(x - 1, y - 1));
							relDistances.put(new Point(x - 1, y - 1), relD);
							image.setRGB(x - 1, y - 1, blue);
						}
						if (image.getRGB(x, y - 1) == BIN1) {
							succ.push(new Point(x, y - 1));
							relDistances.put(new Point(x, y - 1), relD);
							image.setRGB(x, y - 1, blue);
						}
						if (image.getRGB(x + 1, y - 1) == BIN1) {
							succ.push(new Point(x + 1, y - 1));
							relDistances.put(new Point(x + 1, y - 1), relD);
							image.setRGB(x + 1, y - 1, blue);
						}
						if (image.getRGB(x - 1, y) == BIN1) {
							succ.push(new Point(x - 1, y));
							relDistances.put(new Point(x - 1, y), relD);
							image.setRGB(x - 1, y, blue);
						}
						if (image.getRGB(x + 1, y) == BIN1) {
							succ.push(new Point(x + 1, y));
							relDistances.put(new Point(x + 1, y), relD);
							image.setRGB(x + 1, y, blue);
						}
						if (image.getRGB(x - 1, y + 1) == BIN1) {
							succ.push(new Point(x - 1, y + 1));
							relDistances.put(new Point(x - 1, y + 1), relD);
							image.setRGB(x - 1, y + 1, blue);
						}
						if (image.getRGB(x, y + 1) == BIN1) {
							succ.push(new Point(x, y + 1));
							relDistances.put(new Point(x, y + 1), relD);
							image.setRGB(x, y + 1, blue);
						}
						if (image.getRGB(x + 1, y + 1) == BIN1) {
							succ.push(new Point(x + 1, y + 1));
							relDistances.put(new Point(x + 1, y + 1), relD);
							image.setRGB(x + 1, y + 1, blue);
						}
					} else // The red point is within reach
					{
						if (q.equals(redPoint))
							relDistances.put(q, relD);

					}
				}
			}
			pred.addAll(succ); // The successors become the new predecessors
		} while (!succ.empty());

		// Go backwards the path from the red point to from
		int distToFrom = Integer.MAX_VALUE;
		Vector<Point> wayPoints = new Vector<Point>();
		if (relDistances.containsKey(redPoint)) // There is a path between both
		// points, add it to the path
		{
			boolean pointLeft = true;
			Point current = redPoint;
			wayPoints.add(current);

			while (pointLeft) {
				int x = current.x;
				int y = current.y;

				// Moore-neighborship of red point
				Point p1 = new Point(x - 1, y - 1);
				Point p2 = new Point(x, y - 1);
				Point p3 = new Point(x + 1, y - 1);
				Point p4 = new Point(x - 1, y);
				Point p5 = new Point(x + 1, y);
				Point p6 = new Point(x - 1, y + 1);
				Point p7 = new Point(x, y + 1);
				Point p8 = new Point(x + 1, y + 1);
				if ((relDistances.get(p1) != null)
						&& (relDistances.get(p1) < distToFrom)) {
					current = p1;
					wayPoints.add(p1);
				} else if ((relDistances.get(p2) != null)
						&& (relDistances.get(p2) < distToFrom)) {
					current = p2;
					wayPoints.add(p2);
				} else if ((relDistances.get(p3) != null)
						&& (relDistances.get(p3) < distToFrom)) {
					current = p3;
					wayPoints.add(p3);
				} else if ((relDistances.get(p4) != null)
						&& (relDistances.get(p4) < distToFrom)) {
					current = p4;
					wayPoints.add(p4);
				} else if ((relDistances.get(p5) != null)
						&& (relDistances.get(p5) < distToFrom)) {
					current = p5;
					wayPoints.add(p5);
				} else if ((relDistances.get(p6) != null)
						&& (relDistances.get(p6) < distToFrom)) {
					current = p6;
					wayPoints.add(p6);
				} else if ((relDistances.get(p7) != null)
						&& (relDistances.get(p7) < distToFrom)) {
					current = p7;
					wayPoints.add(p7);
				} else if ((relDistances.get(p8) != null)
						&& (relDistances.get(p8) < distToFrom)) {
					current = p8;
					wayPoints.add(p8);
				} else
					pointLeft = false;
				distToFrom = relDistances.get(current);
			}

			if (distToFrom == 0) {
				for (int i = 0; i < wayPoints.size(); i++)
					absDistances.put(wayPoints.elementAt(i), new Integer(
							distanceFromSeed + wayPoints.size() - i));
				return true;
			}
		}
		return false;
	}

	private void investigateMooreNeighborship(BufferedImage image, Point p,
			Stack<Point> succ, HashMap<Point, Integer> dist, int d) {
		int x = p.x, y = p.y;
		int blue = Color.blue.getRGB();

		if (image.getRGB(x - 1, y - 1) == BIN1) {
			succ.push(new Point(x - 1, y - 1));
			dist.put(new Point(x - 1, y - 1), d);
			image.setRGB(x - 1, y - 1, blue);
		}
		if (image.getRGB(x, y - 1) == BIN1) {
			succ.push(new Point(x, y - 1));
			dist.put(new Point(x, y - 1), d);
			image.setRGB(x, y - 1, blue);
		}
		if (image.getRGB(x + 1, y - 1) == BIN1) {
			succ.push(new Point(x + 1, y - 1));
			dist.put(new Point(x + 1, y - 1), d);
			image.setRGB(x + 1, y - 1, blue);
		}
		if (image.getRGB(x - 1, y) == BIN1) {
			succ.push(new Point(x - 1, y));
			dist.put(new Point(x - 1, y), d);
			image.setRGB(x - 1, y, blue);
		}
		if (image.getRGB(x + 1, y) == BIN1) {
			succ.push(new Point(x + 1, y));
			dist.put(new Point(x + 1, y), d);
			image.setRGB(x + 1, y, blue);
		}
		if (image.getRGB(x - 1, y + 1) == BIN1) {
			succ.push(new Point(x - 1, y + 1));
			dist.put(new Point(x - 1, y + 1), d);
			image.setRGB(x - 1, y + 1, blue);
		}
		if (image.getRGB(x, y + 1) == BIN1) {
			succ.push(new Point(x, y + 1));
			dist.put(new Point(x, y + 1), d);
			image.setRGB(x, y + 1, blue);
		}
		if (image.getRGB(x + 1, y + 1) == BIN1) {
			succ.push(new Point(x + 1, y + 1));
			dist.put(new Point(x + 1, y + 1), d);
			image.setRGB(x + 1, y + 1, blue);
		}
	}

	public boolean buildPath(TreeNode firstNode, Vector<TreeNode> nodes,
			HashMap<Point, HashMap<Point, Integer>> distances,
			Vector<Branch> temp) {
		int dist = Integer.MAX_VALUE; // distance to another node is infinite
		Point from = firstNode.getLocation();
		Point current = from;
		Iterator<Point> seeds = distances.keySet().iterator();
		Vector<Point> path = new Vector<Point>(); // Holds the pixels forming a
		// path between two nodes
		path.add(from);

		try {
			while (seeds.hasNext()) // Browse through all hashes, each one
			// contains all paths emerging from {seed}
			{
				Point seed = seeds.next();
				HashMap<Point, Integer> visitedPoints = distances.get(seed); // Points
				// visited
				// starting
				// at
				// point
				// {seed}

				if ((visitedPoints != null)
						&& (visitedPoints.containsKey(from))
						&& (!seed.equals(from))) {

					// Move the path backwards only accepting moves on points
					// that have a lower distance from {seed}
					boolean pointLeft = true;
					while (pointLeft) {
						// Inspect Moore-neighborship
						int x = current.x;
						int y = current.y;

						Point p1 = new Point(x - 1, y - 1);
						Point p2 = new Point(x, y - 1);
						Point p3 = new Point(x + 1, y - 1);
						Point p4 = new Point(x - 1, y);
						Point p5 = new Point(x + 1, y);
						Point p6 = new Point(x - 1, y + 1);
						Point p7 = new Point(x, y + 1);
						Point p8 = new Point(x + 1, y + 1);

						if ((visitedPoints.get(p1) != null)
								&& (visitedPoints.get(p1) < dist))
							current = p1;
						else if ((visitedPoints.get(p2) != null)
								&& (visitedPoints.get(p2) < dist))
							current = p2;
						else if ((visitedPoints.get(p3) != null)
								&& (visitedPoints.get(p3) < dist))
							current = p3;
						else if ((visitedPoints.get(p4) != null)
								&& (visitedPoints.get(p4) < dist))
							current = p4;
						else if ((visitedPoints.get(p5) != null)
								&& (visitedPoints.get(p5) < dist))
							current = p5;
						else if ((visitedPoints.get(p6) != null)
								&& (visitedPoints.get(p6) < dist))
							current = p6;
						else if ((visitedPoints.get(p7) != null)
								&& (visitedPoints.get(p7) < dist))
							current = p7;
						else if ((visitedPoints.get(p8) != null)
								&& (visitedPoints.get(p8) < dist))
							current = p8;
						else
							pointLeft = false;

						path.add(current);
						dist = visitedPoints.get(current);
					}

					if (dist == 0) {
						TreeNode n = topology.getNodeAt(seed);
						TreeNode secondNode = null;
						if (n instanceof InnerNode)
							secondNode = (InnerNode) n;
						else if (n instanceof Tip)
							secondNode = (Tip) n;
						if (!((firstNode instanceof Tip) && (secondNode instanceof Tip))) {
							Branch branch = new Branch(firstNode, secondNode,
									false);
							branch.setLengthInPixels(visitedPoints.get(from));
							branch.setPath(path);
							calculateBranchStraightness(branch);
							temp.add(branch);
							path.clear();
							path.add(from); // Re-initialize path with seed
							// location
						}
						current = from;
						dist = Integer.MAX_VALUE;
					}
				}
			}
		} catch (Exception ex) {
			// The topology could not be built; an error occurred
			return false;
		}
		return true;
	}

	public void identifyInflectionPoints(Vector<Branch> branches,
			Vector<TreeNode> nodes) {
		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch branch = (Branch) it.next();
			if ((wizard.topologyType == RECTANGULAR)
					|| (wizard.topologyType == POLAR))
				calculateChainCode(branch);
		}

		int iNum = 0, oNum = 0;
		float outerNumTurningPoints = 0, innerNumTurningPoints = 0;
		float avInnerNum, avOuterNum = 0;
		Iterator<Branch> bit = branches.iterator();
		Vector<Point> turningPoints;

		while (bit.hasNext()) {
			Branch b = (Branch) bit.next();
			// Vector<Point> path = b.getPath();
			TreeNode firstNode = b.getFirstNode();
			TreeNode secondNode = b.getSecondNode();
			turningPoints = (Vector<Point>) b.getTurningPoints();

			// Delete turning points that are too near a node with respect to
			// the branch length

			Iterator<Point> tp = turningPoints.iterator();
			while (tp.hasNext()) {
				Point p = tp.next();
				if ((p.distance(firstNode.getLocation()) < 2)
						|| (p.distance(secondNode.getLocation()) < 2)) {
					turningPoints.removeElement(p);
				}
			}

			b.setTurningPoints(turningPoints);

			// Get number of turning points for the very branch
			if ((b.getFirstNode() instanceof Tip)
					|| (b.getSecondNode() instanceof Tip)) {
				outerNumTurningPoints += turningPoints.size();
				oNum++;
			} else {
				innerNumTurningPoints += turningPoints.size();
				iNum++;
			}

		}

		avInnerNum = innerNumTurningPoints / iNum;
		avOuterNum = outerNumTurningPoints / oNum;

		// Look if there are pairs of outer branches emerging from a node
		// If the branches have roughly the same size, delete the redundant
		// inflection point(s)
		Iterator<TreeNode> ni = nodes.iterator();
		while (ni.hasNext()) {
			TreeNode n = ni.next();
			if (n instanceof InnerNode) {
				Vector<Branch> outerBranches = topology.getOuterBranches();
				if (outerBranches.size() == 2) // Should be one "ingoing" branch
				// and two outer branches with
				// tips
				{
					Branch b1 = outerBranches.firstElement();
					Branch b2 = outerBranches.lastElement();
					int l1 = b1.getTotalLengthInPixels();
					int l2 = b2.getTotalLengthInPixels();

				}
			}
		}
	}

	public void calculateSegmentSlope(Vector<Branch> branches) {
		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch branch = (Branch) it.next();

			for (int i = 0; i < branch.getNumSegments(); i++) {
				// Get the pixel positions that make the branch segment
				Vector<Point> segment = branch.getPathSegmentPoints(i);

				// Check whether the segment is straight
				double straightness = calculateSegmentStraightness(branch, i);
				branch.setSegmentStraightness(i, straightness);
				// If so, calculate the angle between the segment and a
				// horizontal line
				if ((straightness >= 60) && (segment != null)) // The segment is
				// more or less
				// straight
				{
					double alpha = Double.NaN;
					Point p1 = segment.firstElement();
					Point p2 = segment.lastElement();
					Point A = null, B = null, C = null;

					if ((p1.x <= p2.x) && (p1.y <= p2.y)) {
						A = p2;
						B = p1;
					} else if ((p1.x <= p2.x) && (p2.y <= p1.y)) {
						A = p1;
						B = p2;
					} else if ((p1.x >= p2.x) && (p2.y <= p1.y)) {
						A = p1;
						B = p2;
					} else if ((p1.x >= p2.x) && (p2.y >= p1.y)) {
						A = p2;
						B = p1;
					}

					C = new Point(B.x, A.y); // AC

					// Calculate the lengths of triangle sides
					double c = A.distance(B);
					double b = A.distance(C);
					double a = C.distance(B);

					// Calculate the angle with a horizontal line AC
					alpha = Math.toDegrees(Math.asin(a / c));

					branch.setSegmentSlope(i, alpha);
				} else
					branch.setSegmentSlope(i, Double.NaN);
			}
		}
	}

	public void determineBranchSegments(Vector<Branch> branches,
			Vector<TreeNode> nodes) {
		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch b = it.next();
			Vector<Point> turningPoints = b.getTurningPoints();
			Vector<Point> path = b.getPath();

			// Build path segments
			Point from = path.firstElement();
			Point to = null;

			if (turningPoints.size() == 0) // Branch has no bend, consists of
			// one segment that is the whole
			// branch
			{
				b.addPathSegment(b, path.firstElement(), path.lastElement(),
						NOT_DETERMINED);
			} else // Branch has at least one bend (i.e. is made of at least two
			// segments)
			{
				Iterator<Point> pathIt = path.iterator();
				while (pathIt.hasNext()) {
					Point p = pathIt.next();
					if (turningPoints.contains(p)) // End/begin of a segment
					{
						to = p;
						b.addPathSegment(b, from, to, NOT_DETERMINED);
						from = p;
						turningPoints.remove(p);
					}
				}
				b.addPathSegment(b, to, path.lastElement(), NOT_DETERMINED);
			}
		}
	}

	public void determineLengthRelevantSegments(Vector<Branch> branches,
			Vector<TreeNode> nodes) {
		Vector<Integer> allLengths = new Vector<Integer>();
		Vector<Double> allSlopes = new Vector<Double>();
		Vector<Double> allOuterSlopes = new Vector<Double>();
		Vector<Double> allStraightness = new Vector<Double>();

		Iterator<Branch> it = branches.iterator();
		while (it.hasNext()) {
			Branch b = it.next();
			for (int i = 0; i < b.getNumSegments(); i++) {
				Vector<Double> slopes = b.getSegSlope();
				Vector<Double> straightness = b.getSegStraightness();
				Vector<Integer> lengthInPixels = b.getSegLengthPixels();
				allSlopes.addAll(slopes);
				allStraightness.addAll(straightness);
				allLengths.addAll(lengthInPixels);
				if (b.isOuterBranch()) {
					// Find the slope of the outmost segment
					Tip tip;
					if (b.getFirstNode() instanceof Tip)
						tip = (Tip) b.getFirstNode();
					else
						tip = (Tip) b.getSecondNode();
					Vector<Point> segment = b.getPathSegmentPoints(i);
					if (segment.contains(tip.getLocation())) {
						allOuterSlopes.add(b.getSegSlope().elementAt(i));
					}
				}
			}
		}

		if (wizard.topologyType == FREE) {
			// All branch segments are length relevant
			Iterator<Branch> bt = branches.iterator();
			while (bt.hasNext()) {
				Branch b = bt.next();
				for (int i = 0; i < b.getNumSegments(); i++) {
					b.setSegmentLengthRelevant(i, RELEVANT);
				}
			}
		} else if (wizard.topologyType == RECTANGULAR) {
			// Normally all branch segments that have tips attached are length
			// relevant
			// Has the majority of those segments an identical slope?
			double[] outerSlopes = new double[allOuterSlopes.size()];
			for (int j = 0; j < allOuterSlopes.size(); j++) {
				outerSlopes[j] = allOuterSlopes.elementAt(j);
			}
			Arrays.sort(outerSlopes);

			// Count how many identical slopes there are, if there is a slope
			// that occurs at least in 80 percent of all slopes,
			// then all branch segments in the tree with this slope are regarded
			// as length relevant
			int identical = 0;
			int refSlope = (int) Math.round(outerSlopes[0]);

			// Build a set of bins, each bin contains all nearly identical
			// slopes
			// A new slope becomes the first member of a new bin
			// After that, a slope belongs in a bin if the bin's first member
			// (slope) deviates not more than 3
			HashMap<Integer, Integer> bins = new HashMap<Integer, Integer>();
			bins.put(refSlope, 1);
			for (int i = 1; i < outerSlopes.length; i++) {
				int slope = (int) Math.round(outerSlopes[1]);
				int j = -3;
				boolean binFound = false;
				do {
					if (bins.containsKey(slope + j)) {
						bins.put(slope + j, (int) (bins.get(slope + j)) + 1);
						binFound = true;
					}
					j++;
				} while ((!binFound) && (j < 6));
				if (!binFound)
					bins.put(slope, 1);
			}

			// Find the bin with most members
			Set<Integer> keys = bins.keySet();
			int theValue = -1;
			Iterator<Integer> keyIt = keys.iterator();
			while (keyIt.hasNext()) {
				int key = keyIt.next();
				int amount = bins.get(key);

				if (amount > identical) {
					theValue = key;
					identical = amount;
				}
			}

			// The branch segments that have the same slopes as the members in
			// this bin
			// are set length relevant, the others are set length irrelevant
			int compSlope = theValue;

			if (((float) identical / (float) outerSlopes.length) >= 0.8) {
				Iterator<Branch> bt = branches.iterator();
				while (bt.hasNext()) {
					Branch b = bt.next();
					for (int i = 0; i < b.getNumSegments(); i++) {
						Vector<Double> slopes = b.getSegSlope();
						if (Math.abs(slopes.elementAt(i) - compSlope) < 10) {
							b.setSegmentLengthRelevant(i, RELEVANT);
						} else
							b.setSegmentLengthRelevant(i, IRRELEVANT);
					}
				}

			}
		} else if (wizard.topologyType == POLAR) {

			// The outmost segment of an outer branch is length relevant, the
			// second is not, etc.
			Iterator<Branch> bt = branches.iterator();
			while (bt.hasNext()) {
				Branch b = bt.next();
				if (b.isOuterBranch) {
					Tip tip;
					if (b.getFirstNode() instanceof Tip)
						tip = (Tip) b.getFirstNode();
					else
						tip = (Tip) b.getSecondNode();
					b.orderSegmentsRelativeTo(tip);
					Vector<Integer> segOrder = b.getSegOrder();
					int rel = RELEVANT;

					for (int i = 0; i < segOrder.size(); i++) {
						b.setSegmentLengthRelevant(segOrder.elementAt(i), rel);
						if (rel == RELEVANT)
							rel = IRRELEVANT;
						else
							rel = RELEVANT;
					}

					// Those outer branches that have only one segment and end
					// at the same inner node
					// should be treated differently
				}
			}

			// Build a container with inner nodes that have a Tip as neighbor
			Vector<InnerNode> innerNodes = topology.getInnerNodes();
			Vector<InnerNode> nodesToInvestigate = new Vector<InnerNode>();
			Vector<InnerNode> nextNodes = new Vector<InnerNode>();

			for (int i = 0; i < innerNodes.size(); i++) {
				boolean hasTipAsNeighbor = false;
				InnerNode innerNode = innerNodes.elementAt(i);
				Vector<TreeNode> neighbors = innerNode.getNeighbors();
				for (int j = 0; j < neighbors.size(); j++) {
					if (neighbors.elementAt(j) instanceof Tip)
						hasTipAsNeighbor = true;
				}
				if (hasTipAsNeighbor)
					nodesToInvestigate.add(innerNode);
			}

			int neighborsProcessed = 0;

			Iterator<InnerNode> nt = nodesToInvestigate.iterator();
			do {
				do {
					// Inspect the branches of the next inner node
					InnerNode innerNode = nt.next();
					Vector<Branch> itsBranches = topology
							.getBranchesAtNode(innerNode);
					neighborsProcessed = 0;
					Branch branch = null, missingBranch = null;

					for (int i = 0; i < itsBranches.size(); i++) {
						branch = itsBranches.elementAt(i);

						// Count number of processed branches linked to this
						// inner node
						Vector<Integer> segLengthRelevant = branch
								.getSegLengthRelevant();
						// Find out whether this branch has aready been
						// processed
						if (segLengthRelevant.elementAt(0) != NOT_DETERMINED)
							neighborsProcessed++; // Processed
						else
							missingBranch = branch; // Not processed
					}

					// If (n-1) of it branches have been marked before, treat
					// branch n like this:
					// mark the next portion of branch n as length relevant, the
					// next as irrelevant etc.
					if (neighborsProcessed == itsBranches.size() - 1) {
						int rel = RELEVANT;
						missingBranch.orderSegmentsRelativeTo(innerNode);
						Vector<Integer> segOrder = missingBranch.getSegOrder();
						Vector<Double> segStraightness = missingBranch
								.getSegStraightness();

						for (int i = 0; i < segOrder.size(); i++) {
							if ((rel == RELEVANT)
									&& (segStraightness.elementAt(segOrder
											.elementAt(i)) >= 60)) {
								missingBranch.setSegmentLengthRelevant(segOrder
										.elementAt(i), rel);
							} else
								missingBranch.setSegmentLengthRelevant(segOrder
										.elementAt(i), IRRELEVANT);
							if (rel == RELEVANT)
								rel = IRRELEVANT;
							else
								rel = RELEVANT;
						}

						// All branches that are linked to this node have been
						// processed;
						nodesToInvestigate.remove(innerNode);
						nt = nodesToInvestigate.iterator();

						// The other inner node of this branch has to be
						// processed in the next round
						if (missingBranch.getFirstNode() == innerNode) {
							nextNodes.add((InnerNode) missingBranch
									.getSecondNode());
						} else
							nextNodes.add((InnerNode) missingBranch
									.getFirstNode());
					}
				} while (nt.hasNext());
				System.out.println("First bunch processed");
				return;
				// The branches at all inner nodes looked at so far have been
				// processsed
				// Now, investigate the inner nodes at the other side of these
				// branches
				// nodesToInvestigate.addAll(nextNodes);
				// nt = nodesToInvestigate.iterator();
				// nextNodes.clear();
			} while (!nodesToInvestigate.isEmpty());
		}
	}

	// Not currently used
	public void determineLengthRelevantBranchPortions(Vector<Branch> branches,
			Vector<TreeNode> nodes) {
		// Separate inner nodes from tips
		Vector<InnerNode> innerNodes = new Vector<InnerNode>();
		Vector<Tip> tips = new Vector<Tip>();
		for (int i = 0; i < nodes.size(); i++) {
			TreeNode n = nodes.elementAt(i);
			if (n instanceof InnerNode)
				innerNodes.add((InnerNode) n);
			else
				tips.add((Tip) n);
		}

		// Segment the outer branches into length-relevant and irrelevant
		// portions
		for (int i = 0; i < branches.size(); i++) {
			Branch b = branches.elementAt(i);
			if (b.isOuterBranch()) // One node of b is a tip
			{
				Vector<Point> turningPoints = b.getTurningPoints();
				int num = turningPoints.size();
				Vector<Point> path = b.getPath();

				if (num == 0) {

					// No inflection point, the branch in its entirety is length
					// relevant
					Point start = path.firstElement();
					Point end = path.lastElement();
					b.addPathSegment(b, start, end, RELEVANT);
				} else if (num == 1) // One inflection point; the branch
				// consists of two sections
				{
					// The first section is relevant for the branch length, the
					// second is not
					Point start, middle = turningPoints.firstElement(), end;
					Point firstPathLocation = path.firstElement();
					Point lastPathLocation = path.lastElement();
					Point firstNodeLocation = b.getFirstNode().getLocation();

					if (firstPathLocation.equals(firstNodeLocation)) {
						start = firstPathLocation;
						end = lastPathLocation;
					} else {
						start = lastPathLocation;
						end = firstPathLocation;
					}

					b.addPathSegment(b, start, middle, RELEVANT);
					b.addPathSegment(b, middle, end, IRRELEVANT);
				} else // Two or more inflection points; the branch consists of
				// at least three sections
				{
					Tip tip;
					InnerNode innerNode;
					if (b.getFirstNode() instanceof Tip) {
						tip = (Tip) b.getFirstNode();
						innerNode = (InnerNode) b.getSecondNode();
					} else {
						tip = (Tip) b.getSecondNode();
						innerNode = (InnerNode) b.getFirstNode();
					}
					;

					Point firstPathLocation = path.firstElement();
					Point lastPathLocation = path.lastElement();
					Point start, end;
					int lengthRelevant = RELEVANT;

					if (firstPathLocation.equals(tip.getLocation())) // Inflection
					// points
					// and
					// path
					// points
					// sorted
					// in
					// same
					// order
					{
						System.out.println("forward");
						start = firstPathLocation;
						end = turningPoints.elementAt(0);
						b.addPathSegment(b, start, end, lengthRelevant);

						for (int j = 1; j < turningPoints.size(); j++) {
							if (lengthRelevant == RELEVANT)
								lengthRelevant = IRRELEVANT;
							else
								lengthRelevant = RELEVANT;
							start = turningPoints.elementAt(j - 1);
							end = turningPoints.elementAt(j);
							b.addPathSegment(b, start, end, lengthRelevant);
						}
						if (lengthRelevant == RELEVANT)
							lengthRelevant = IRRELEVANT;
						else
							lengthRelevant = RELEVANT;
						b.addPathSegment(b, end, innerNode.getLocation(),
								lengthRelevant);

					} else

					{

						System.out.println("reverse");
						start = lastPathLocation;
						end = path.elementAt(path.size() - 1);

						for (int j = turningPoints.size(); j > 1; j--) {
							if (lengthRelevant == RELEVANT)
								lengthRelevant = IRRELEVANT;
							else
								lengthRelevant = RELEVANT;
							b.addPathSegment(b, start, end, lengthRelevant);
							start = turningPoints.elementAt(j);
							end = turningPoints.elementAt(j - 1);
						}
						b.addPathSegment(b, end, path.firstElement()
								.getLocation(), lengthRelevant);
					}
				}

				// Store the inner node of b
				TreeNode n1 = b.getFirstNode();
				TreeNode n2 = b.getSecondNode();
				if (n1 instanceof InnerNode)
					innerNodes.add((InnerNode) n1);
				else
					innerNodes.add((InnerNode) n2);
			}

		} // end for

		// Segment the inner branches into length-relevant and irrelevant
		// portions
		// Start at the inner nodes that belong to the outer branches
		Iterator<InnerNode> it = innerNodes.iterator();
		do {
			// Inspect the next inner node
			InnerNode in = it.next();
			// n is the number of branches that emerge from in
			Vector<Branch> itsBranches = topology.getBranchesAtNode(in);
			// If (n-1) of it branches have been marked before, treat branch n
			// like this:
			// mark the next portion of branch n as length relevant, the next as
			// irrelevant etc.
			// Remove in as it has been processed
			innerNodes.remove(in);
			// Put the other node of branch n in innerNodes

		}

		while (!innerNodes.isEmpty());
	}

	public void calculateChainCode(Branch branch) {
		Vector<Point> path = branch.getPath();
		Vector<Integer> chain = getChainCode(path);
		Vector<Point> positions = calculateTurningPointCandidates(branch, chain);
		branch.setTurningPoints(positions);
	}

	public double calculateSegmentStraightness(Branch branch, int segNum) {
		Vector<Point> segment = branch.getPathSegmentPoints(segNum);
		int segmentLength = segment.size();
		Point p = null;
		int h = 0;
		int cnt = 0;

		int x1 = segment.firstElement().x;
		int y1 = segment.firstElement().y;
		int x2 = segment.lastElement().x;
		int y2 = segment.lastElement().y;
		Line2D.Float idealLine = new Line2D.Float(x1, y1, x2, y2);

		// Ignore 5% of the path pixels at each side of the path
		int ignoreNum = 0;
		if (segmentLength > 5)
			ignoreNum = (int) (segmentLength / 20);
		else
			ignoreNum = 1;
		// Is there a point from the Moore-neighborship of each path-point on
		// the
		// ideal line between the first and the last path point?
		Iterator<Point> it = segment.iterator();
		while (it.hasNext()) {
			cnt++;
			p = it.next();
			float x = p.x;
			float y = p.y;
			if ((cnt > ignoreNum) && (cnt < segmentLength - ignoreNum)) {
				if (idealLine.ptLineDist(x, y) < 3.0)
					h++;
			}
		}
		double percentage = 100 * ((double) h)
				/ ((double) segmentLength - 2 * ignoreNum);

		return percentage;
	}

	public void calculateBranchStraightness(Branch branch) {

		Vector<Point> path = branch.getPath();
		int pathLength = path.size();
		Point p = null;
		int h = 0;
		int cnt = 0;

		int x1 = branch.getFirstNode().getX();
		int y1 = branch.getFirstNode().getY();
		int x2 = branch.getSecondNode().getX();
		int y2 = branch.getSecondNode().getY();

		Line2D.Float idealLine = new Line2D.Float(x1, y1, x2, y2);

		// Ignore 5% of the path pixels at each side of the path
		int ignoreNum = 0;
		if (pathLength > 5)
			ignoreNum = (int) (pathLength / 20);
		else
			ignoreNum = 1;
		// Is there a point from the Moore-neighborship of each path-point on
		// the
		// ideal line between the first and the last path point?
		Iterator<Point> it = path.iterator();
		while (it.hasNext()) {
			cnt++;
			p = it.next();
			float x = p.x;
			float y = p.y;
			if ((cnt > ignoreNum) && (cnt < pathLength - ignoreNum)) {
				if (idealLine.ptLineDist(x, y) < 3.0)
					h++;
			}
		}
		double percentage = 100 * ((double) h)
				/ ((double) pathLength - 2 * ignoreNum);
		branch.setStraightness(percentage);
	}

	private Vector<Integer> getChainCode(Vector<Point> path) {
		Vector<Integer> chain = new Vector<Integer>();
		Iterator<Point> it = path.iterator();
		Point p1 = it.next();
		while (it.hasNext()) {
			Point p2 = it.next();
			chain.add(Direction.getBearing(p1, p2));
			p1 = p2; // The current point becomes the old point
		}
		chain.removeElement(chain.lastElement());
		return chain;
	}

	// Calculates turning points based on the chain code for a path
	private Vector<Point> calculateTurningPointCandidates(Branch branch,
			Vector<Integer> chain) {
		BufferedImage image = imageBuffer.getProcessedImage();
		Vector<Point> turningPoints = new Vector<Point>();
		Vector<Boolean> containsCorner = new Vector<Boolean>(chain.size());
		int cbearing;
		int bearing = chain.elementAt(0);
		Vector<Point> path = branch.getPath();
		boolean overlap = false;

		// No portion of the path has been inspected so far
		for (int k = 0; k < chain.size(); k++)
			containsCorner.add(new Boolean(false));

		int i;
		int j;
		int step = 2;
		int offset = 1; // Omit offset pixels at the start and the end of the
		// chain

		if (chain.size() > 2) {
			for (step = 2; step < 10; step++) // Max. segment size
			{
				i = offset;
				j = i + step;

				while (j < chain.size() - offset) {
					// Analyze path segment [i..j]
					bearing = chain.elementAt(i);
					cbearing = chain.elementAt(j);

					// Check on overlap between path segment [i..j] and an
					// already inspected path segment
					int p = i;
					overlap = false;
					while (p < j) {
						if (containsCorner.elementAt(p) == true) {
							overlap = true;
							break;
						}
						p++;
					}

					if (!overlap) // No overlap
					{
						if (((bearing == 0) && ((cbearing == 7)
								|| (cbearing == 0) || (cbearing == 1)))
								|| ((bearing == 1) && ((cbearing == 0)
										|| (cbearing == 1) || (cbearing == 2)))
								|| ((bearing == 2) && ((cbearing == 1)
										|| (cbearing == 2) || (cbearing == 3)))
								|| ((bearing == 3) && ((cbearing == 2)
										|| (cbearing == 3) || (cbearing == 4)))
								|| ((bearing == 4) && ((cbearing == 3)
										|| (cbearing == 4) || (cbearing == 5)))
								|| ((bearing == 5) && ((cbearing == 4)
										|| (cbearing == 5) || (cbearing == 6)))
								|| ((bearing == 6) && ((cbearing == 5)
										|| (cbearing == 6) || (cbearing == 7)))
								|| ((bearing == 7) && ((cbearing == 6)
										|| (cbearing == 7) || (cbearing == 0)))) {
							i++;
							j = i + step;
						} else {
							int k = Math.abs(i + (j - i) / 2); // Index position
							// of the
							// putative
							// extremum
							if (checkRightAngle(path.elementAt(k), image)) {
								turningPoints.add(path.elementAt(k));

								// Mark the path segment [i..j] that contains
								// the inflection point
								for (int l = i; l < j; l++)
									containsCorner.set(l, true);

								// Move the interval to [j..i+step] (the first
								// path position after the interval in which the
								// turning point was found
								i = j;
								j = i + step;
							} else {
								i++;
								j = i + step;
							}
						}
					} // end if
					else // Overlap
					{
						i = p + 1;
						j = i + step;
					}
				} // end while
			} // end for
		} // end if

		return turningPoints;
	}

	private boolean checkRightAngle(Point midpoint, BufferedImage image) {
		try {
			// Given the sides a, b, c of a triangle
			// m is the point in which the sides b and c meet
			// alpha is the angle between b and c
			// a is the third side
			// Check whether alpha is an approximate right angle
			// Alpha is defined by two foreground paths emerging from m
			BufferedImage portion = getClonedImage(image);
			// boolean crossingReached = false;
			boolean crossingReachedLeft = false, crossingReachedRight = false;
			Point next = null;
			double a = 1, b = 1, c = 1;
			double alpha;
			int x1 = midpoint.x, y1 = midpoint.y;
			int x2 = midpoint.x, y2 = midpoint.y;
			int green = Color.green.getRGB();
			double sum = 0;
			int n = 0;
			int length = 1;

			// Mark the midpoint as visited;
			portion.setRGB(midpoint.x, midpoint.y, green);

			// Both pathways emerge at midpoint
			Point start1 = getN8PointWithColor(portion, new Point(midpoint.x,
					midpoint.y), BIN1);
			x1 = start1.x;
			y1 = start1.y;
			portion.setRGB(start1.x, start1.y, green);
			Point start2 = getN8PointWithColor(portion, new Point(midpoint.x,
					midpoint.y), BIN1);
			x2 = start2.x;
			y2 = start2.y;
			portion.setRGB(start2.x, start2.y, green);

			do {
				// Follow first path starting from the midpoint
				// Move to the next path point unless it is a crossing
				if (crossingReachedLeft == false) {
					if (getN8FGCnt(portion, new Point(x1, y1), BIN1) == 1) {
						next = getN8PointWithColor(portion, new Point(x1, y1),
								BIN1);
						x1 = next.x;
						y1 = next.y;
						portion.setRGB(x1, y1, green);
					} else
						crossingReachedLeft = true;
				}

				// Follow second path starting from the midpoint
				// Move to the next path point unless it is a crossing
				if (crossingReachedRight == false) {
					if (getN8FGCnt(portion, new Point(x2, y2), BIN1) == 1) {
						next = getN8PointWithColor(portion, new Point(x2, y2),
								BIN1);
						x2 = next.x;
						y2 = next.y;
						portion.setRGB(x2, y2, green);
					} else
						crossingReachedRight = true;
				}

				// Calculate the lengths of triangle sides a, b and c
				b = 10 * new Point(x1, y1).distance(midpoint);
				c = 10 * new Point(x2, y2).distance(midpoint);
				a = 10 * new Point(x1, y1).distance(new Point(x2, y2));

				// Calculate the angle alpha at the putative inflection point
				alpha = Math.toDegrees(Math.acos((b * b + c * c - a * a)
						/ (2 * b * c)));

				// Allow only calculations for branches with a significant
				// length
				if ((length > 0) && (alpha != Double.NaN)) {
					sum = sum + length * alpha;
					n = n + length;
				}

				// Travel further down the emerging paths
				length++;
			} while (length < 10);

			int avAngle = (int) Math.abs(sum / n);

			if ((avAngle > 20) && (avAngle < 160))
				return true;
		} catch (Exception ex) {
		}
		;
		return false;
	}

	private BufferedImage getClonedImage(BufferedImage img) {
		String[] pnames = img.getPropertyNames();
		Hashtable<String, Object> cproperties = new Hashtable<String, Object>();
		if (pnames != null) {
			for (int j = 0; j < pnames.length; j++) {
				cproperties.put(pnames[j], img.getProperty(pnames[j]));
			}
		}
		WritableRaster wr = img.getRaster();
		WritableRaster cwr = wr.createCompatibleWritableRaster();
		cwr.setRect(wr);

		return new BufferedImage(img.getColorModel(), cwr, img
				.isAlphaPremultiplied(), cproperties);
	}

	public void setObjectReferences(Wizard wizard, TreeTopology topology) {
		this.wizard = wizard;
		this.topology = topology;
	}

}

final class Direction {
	public Direction() {
	};

	public static int getXOffset(int direction) {
		if ((direction == 0) || (direction == 1) || (direction == 7))
			return 1;
		if ((direction == 2) || (direction == 6))
			return 0;
		if ((direction == 3) || (direction == 4) || (direction == 5))
			return -1;
		return 2; // Error
	}

	public static int getYOffset(int direction) {
		if ((direction == 1) || (direction == 2) || (direction == 3))
			return 1;
		if ((direction == 0) || (direction == 4))
			return 0;
		if ((direction == 5) || (direction == 6) || (direction == 7))
			return -1;
		return 2; // Error
	}

	public static int getDirection(int xOffset, int yOffset) {
		if ((xOffset == 1) && (yOffset == 0))
			return 0;
		if ((xOffset == 1) && (yOffset == 1))
			return 1;
		if ((xOffset == 0) && (yOffset == 1))
			return 2;
		if ((xOffset == -1) && (yOffset == 1))
			return 3;
		if ((xOffset == -1) && (yOffset == 0))
			return 4;
		if ((xOffset == -1) && (yOffset == -1))
			return 5;
		if ((xOffset == 0) && (yOffset == -1))
			return 6;
		if ((xOffset == 1) && (yOffset == -1))
			return 7;
		return 8; // Error
	}

	public static int getBearing(Point p, Point q) // Returns the direction q
	// has relative to p
	{
		int dx = q.x - p.x;
		int dy = q.y - p.y;
		return getDirection(dx, dy);
	}

	public static int getFitsDirectionMeasure(Point c, Point p, int d) {
		// c is the point in the center of an 8-neighborship
		// p is a point within the 8-neighborship of c
		// d is the preferred direction
		// How well is the spatial relationship between c and p represented by
		// d?
		// There are measures 0 - match, 1 - 45 aberration, ..., 4 - 180
		// aberration
		int oX = (int) (p.getX() - c.getX());
		int oY = (int) (p.getY() - c.getY());
		int dX = getXOffset(d);
		int dY = getYOffset(d);
		int diffX = Math.abs(oX - dX);
		int diffY = Math.abs(oY - dY);
		if ((diffX == 0) && (diffY == 0))
			return 0;
		if (((diffX == 1) && (diffY == 0)) || (((diffX == 0) && (diffY == 1))))
			return 1;
		if ((diffX == 1) && (diffY == 1))
			return 2;
		if (((diffX == 2) && (diffY == 1)) || (((diffX == 1) && (diffY == 2))))
			return 3;
		if ((diffX == 2) && (diffY == 0))
			return 4;
		return 5; // Invalid measure
	}
} // class Direction