/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.Point;
import java.awt.Shape;
import java.awt.geom.Line2D;
import java.io.Serializable;
import java.text.NumberFormat;
import java.util.Iterator;
import java.util.Locale;
import java.util.Vector;

import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.Utils.NumberUtility;

public class Branch implements Serializable, Constants {
	private static final long serialVersionUID = 1L;
	public static int num = 0;
	public static int signalBranch = -1;
	public static int maxFractionDigits = NumberUtility
			.getCurrentNumberFormat().getMaximumFractionDigits();
	private TreeNode firstNode;
	private TreeNode secondNode;
	int totalLengthInPixels = 0;
	double length = defaultBranchLength;
	double userLength = defaultBranchLength;
	boolean userGenerated = true;
	boolean userAssignedLength = false;
	boolean isOuterBranch;
	transient boolean visited = false;
	final int id;
	private NumberFormat nf = NumberFormat.getInstance(Locale.US);
	private double straightness;
	private Vector<Point> path = new Vector<Point>();
	private Vector<Point> turningPoints = new Vector<Point>();
	private Vector<Point> segStart = new Vector<Point>();
	private Vector<Point> segEnd = new Vector<Point>();
	private Vector<Integer> segLengthRelevant = new Vector<Integer>();
	private Vector<Integer> segLengthPixels = new Vector<Integer>();
	private Vector<Double> segStraightness = new Vector<Double>();
	private Vector<Double> segSlope = new Vector<Double>();
	private Vector<Integer> segOrder = new Vector<Integer>(); // Order of the
	// segment with
	// respect to a
	// specified
	// TreeNode

	private int numSegments = 0;

	public Branch(TreeNode node1, TreeNode node2, boolean b) {
		this.id = num++;
		this.firstNode = node1;
		this.secondNode = node2;
		this.userGenerated = b;
	}

	public TreeNode getFirstNode() {
		return this.firstNode;
	}

	public TreeNode getSecondNode() {
		return this.secondNode;
	}

	public boolean getUserGenerated() {
		return this.userGenerated;
	}

	public boolean isUserAssignedLength() {
		return this.userAssignedLength;
	}

	public int getId() {
		return this.id;
	};

	public void setStraightness(double percentage) // Stores how "straight" a
	// line is
	{
		this.straightness = percentage;
	}

	public void setOuterBranch(boolean b) {
		this.isOuterBranch = b;
	}

	public double getStraightness() {
		return this.straightness;
	}

	public Shape getShape() {
		Shape shape = null;
		if ((firstNode != null) && (secondNode != null)) {
			shape = new Line2D.Float(firstNode.getLocation(), secondNode
					.getLocation());
		}
		return shape;
	}

	public void setFirstNode(TreeNode firstNode) {
		this.firstNode = firstNode;
	}

	public void setSecondNode(TreeNode secondNode) {
		this.secondNode = secondNode;
	}

	public void setUserAssignedLength(double length) {
		this.userLength = length;
		this.userAssignedLength = true;
	}

	public void setUserGenerated(boolean b) {
		this.userGenerated = b;
	}

	public void setLengthInPixels(int length) {
		this.totalLengthInPixels = length;
	}

	public void setVisited(boolean b) {
		this.visited = b;
	}

	public boolean getVisited() {
		return this.visited;
	}

	public Vector<Point> getPath() {
		return this.path;
	}

	public boolean isOuterBranch() {
		return this.isOuterBranch;
	}

	public double getLength() {
		if (Wizard.useUserAssignedLengths)
			return userLength; // Only use what the user has typed in
		else if (Wizard.useCalculatedLengths) {
			if (this.getComposedLengthInPixels() > 0) {
				if ((Wizard.manualScaleBarLength == 0.0d)
						|| (Wizard.scaleBarLengthInPixels == 0))
					return this.getComposedLengthInPixels();
				else if ((Wizard.manualScaleBarLength > 0.0d)
						&& (Wizard.scaleBarLengthInPixels > 0))
					return calculateLength();
			} else
				return defaultBranchLength;
		} else if (Wizard.useMixedLengths) {
			if (this.userAssignedLength)
				return userLength;
			else if (this.getComposedLengthInPixels() == 0)
				return defaultBranchLength;
			else
				return calculateLength();
		}

		return defaultBranchLength;
	}

	public double calculateLength() {
		if ((Wizard.scaleBarLengthInPixels == 0)
				|| (Wizard.manualScaleBarLength == 0.0d))
			return getComposedLengthInPixels();
		double factor = Wizard.manualScaleBarLength
				/ Wizard.scaleBarLengthInPixels;
		double scaledLength = this.getComposedLengthInPixels() * factor;
		int maxFractDigits = NumberUtility.getNumFractionDigits(scaledLength);
		if (maxFractDigits > defaultMaxNumFractionDigits) {
			NumberUtility.setMaxNumFractionDigits(defaultMaxNumFractionDigits);
			if (nf != null)
				nf.setMaximumFractionDigits(defaultMaxNumFractionDigits);
		}
		return scaledLength;
	}

	// Returns the length the user has manually assigned to this branch
	// This value is independent from whatever segment composition for the
	// branch
	public double getUserAssignedLength() {
		return userLength;
	}

	// Returns the amount of pixels on the path between both nodes
	public int getTotalLengthInPixels() {
		return totalLengthInPixels;
	}

	// Returns the amount of pixels on the length relevant branch segment(s)
	// If there is only one segment, the method returns the same value as
	// getTotalLengthInPixels()
	public int getComposedLengthInPixels() {
		int lengthPixels = 0;
		for (int i = 0; i < segLengthPixels.size(); i++) {
			if (segLengthRelevant.elementAt(i) == RELEVANT) {
				lengthPixels = lengthPixels
						+ (Integer) segLengthPixels.elementAt(i);
			}
		}

		// The path segments first and last positions overlap; subtract them
		lengthPixels = lengthPixels - (numSegments - 1);

		return lengthPixels;
	}

	public Vector<Point> getSegStart() {
		return this.segStart;
	}

	public Vector<Point> getSegEnd() {
		return this.segEnd;
	}

	public Vector<Integer> getSegLengthRelevant() {
		return this.segLengthRelevant;
	}

	public Vector<Integer> getSegLengthPixels() {
		return this.segLengthPixels;
	}

	public Vector<Double> getSegStraightness() {
		return this.segStraightness;
	}

	public Vector<Double> getSegSlope() {
		return this.segSlope;
	};

	public Vector<Integer> getSegOrder() {
		return this.segOrder;
	};

	public void setPath(Vector<Point> path) {
		for (int i = 0; i < path.size(); i++) {
			this.path.add(new Point(path.elementAt(i)));
		}
	}

	public void addPathSegment(Branch b, Point from, Point to,
			int lengthRelevant) {
		segStart.add(from);
		segEnd.add(to);
		segLengthRelevant.add(lengthRelevant);
		segStraightness.add(new Double(Double.NaN));
		segSlope.add(new Double(Double.NaN));
		// Calculate the length in pixels
		int fromIndex = path.indexOf(from);
		int toIndex = path.indexOf(to);
		int lengthInPixels = Math.abs(toIndex - fromIndex) + 1;
		segLengthPixels.add(lengthInPixels);

		numSegments++;
	}

	// This method does not physically exchange the segment's positions in their
	// container
	// Instead it produces a list of the desired order
	public void orderSegmentsRelativeTo(TreeNode node) {
		// Order the segments this branch is made of with respect to the
		// TreeNode specified
		// The first segment will be that with the TreeNode location as first
		// point
		segOrder = new Vector<Integer>();
		Point location = node.getLocation();

		// If the first segment starts or ends at the node location, keep the
		// segment order
		// otherwise reverse it
		if (numSegments > 1) // Of course there should be more than one segment
		{
			Point firstPoint = segStart.elementAt(0);
			Point lastPoint = segEnd.elementAt(0);
			if ((location.equals(firstPoint)) || (location.equals(lastPoint))) {
				for (int i = 0; i < numSegments; i++)
					segOrder.add(new Integer(i));
			} else {
				for (int i = numSegments; i > 0; i--)
					segOrder.add(new Integer(i - 1));
			}
		} else
			segOrder.add(new Integer(0)); // There is only one segment
	}

	public int getNumSegments() {
		return this.segStart.size();
	}

	public Vector<Point> getPathSegmentPoints(int whichSegment) {
		if (path != null) {
			Vector<Point> points = new Vector<Point>();
			int fromIndex = path.indexOf(segStart.elementAt(whichSegment));
			int toIndex = path.indexOf(segEnd.elementAt(whichSegment));

			if ((fromIndex != -1) && (toIndex != -1)) {
				if (fromIndex <= toIndex) {
					for (int i = fromIndex; i < toIndex; i++)
						points.add(path.elementAt(i));
				} else {
					for (int i = toIndex; i < fromIndex; i++)
						points.add(path.elementAt(i));
				}
				return points;
			}
		}

		return null;

	}

	public void setSegmentSlope(int num, double slope) {
		this.segSlope.set(num, new Double(Math.round(slope)));
	}

	public void setSegmentStraightness(int num, double percentage) {
		this.segStraightness.set(num, new Double(percentage));
	}

	public void setSegmentLengthRelevant(int num, Integer b) {

		for (int i = 0; i < getNumSegments(); i++) {
			Point first = segStart.elementAt(i);
			Point second = segEnd.elementAt(i);
		}

		try {
			this.segLengthRelevant.set(num, b);
		} catch (IndexOutOfBoundsException e) {
			System.out.print("	Fhrt zu Error\n");
		}
	}

	public void setTurningPoints(Vector<Point> turningPoints) {
		this.turningPoints = turningPoints;
	}

	public Vector<Point> getTurningPoints() {
		return this.turningPoints;
	}

	public String toString() {
		nf = NumberFormat.getInstance(Locale.US);
		nf.setGroupingUsed(false);

		String t1 = "", t2 = "";
		String s = "Branch from ";
		TreeNode first = this.getFirstNode();
		TreeNode second = this.getSecondNode();
		int x1 = -1, x2 = -1, y1 = -1, y2 = -1;
		if (first != null) {
			x1 = first.getX();
			y1 = first.getY();
			if (first instanceof Tip)
				t1 = "Tip";
			else if (first instanceof InnerNode)
				t1 = "Inner Node";
			else
				t1 = "Unknown";
		}
		if (second != null) {
			x2 = second.getX();
			y2 = second.getY();
			if (second instanceof Tip)
				t2 = "Tip";
			else if (second instanceof InnerNode)
				t2 = "Inner Node";
			else
				t2 = "Unknown";
		}

		s = s + t1 + " at (" + x1 + ", " + y1 + ") to " + t2 + " at (" + x2
				+ ", " + y2 + ")";
		if (this.totalLengthInPixels > 0) {
			s = s + " is " + this.totalLengthInPixels + " pixels long";
			s = s + ", has calculated length " + nf.format(calculateLength());
		} else
			s = s + ", length unknown";

		if (this.userAssignedLength) {
			nf.setMinimumFractionDigits(NumberUtility
					.getNumFractionDigits(this.userLength));
			s = s + ", has user-assigned length " + nf.format(this.userLength);
		}

		// Straightness of the entire branch
		s = s + ", " + getStraightnessString(this.straightness);

		// Straightness of the segments the branch is made of
		String seg = ", consists of " + numSegments + " line segment(s)";
		s = s + seg;

		return s;
	}

	public String getStraightnessString(double straightness) {
		String s = "", adj = "";
		straightness = straightness * 100 / 100;
		if (straightness < 60)
			adj = " is bent";
		else if ((straightness >= 60) && (straightness < 80))
			adj = " is slightly straight ";
		else if ((straightness >= 80) && (straightness < 90))
			adj = " is reasonably straight";
		else if ((straightness >= 90) && (straightness < 95))
			adj = " is fairly straight ";
		else if ((straightness >= 95) && (straightness < 99.8))
			adj = " is almost straight";
		else if (straightness >= 99.8)
			adj = " is straight, ";
		else
			return " is of unknown gestalt";
		s = s + adj;
		return s;
	}

	public boolean equals(Object o) {
		if (!(o instanceof Branch))
			return false;
		Branch b = (Branch) o;
		TreeNode t1 = b.getFirstNode();
		TreeNode t2 = b.getSecondNode();

		if ((firstNode.equals(t1) && (secondNode.equals(t2))))
			return true;
		if ((secondNode.equals(t1) && (firstNode.equals(t2))))
			return true;
		return false;
	}
}
