/*
    TreeSnatcher Plus - A Phylogenetic Tree Capturing Tool
    Copyright (C) 2010 Thomas Laubach

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package TreeSnatcher.Core;

import java.awt.Rectangle;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.Stack;
import java.util.Vector;

import javax.imageio.ImageIO;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import TreeSnatcher.GUI.GUIActions;
import TreeSnatcher.GUI.ImageBuffer;
import TreeSnatcher.GUI.ImagePanel;
import TreeSnatcher.GUI.Wizard;
import TreeSnatcher.Core.ImageOperations;
import TreeSnatcher.Utils.NumberUtility;
import TreeSnatcher.Utils.Preferences;

public class ActivityProtocol {
	private Stack<Activity> protocol;
	private ImagePanel imagePanel;
	private ImageBuffer imageBuffer;
	private ImageOperations imageOperations;
	private Wizard wizard;
	private GUIActions guiActions;
	private TreeTopology topology;
	private BufferedImage portion = null;
	protected String filename;

	public ActivityProtocol(ImagePanel ip, ImageBuffer ib, ImageOperations io,
			Wizard wiz, GUIActions ga, TreeTopology tt) {
		imagePanel = ip;
		imageBuffer = ib;
		imageOperations = io;
		wizard = wiz;
		guiActions = ga;
		topology = tt;

		protocol = new Stack<Activity>();
	}

	// Not currently used:
	// Returns the sum of all manipulated areas combined for all activities with
	// id {activity}
	// For example: All sections in the image that have been binarized up to now
	public Rectangle2D getCoveredArea(int id) {
		Rectangle2D coveredArea = null;
		Activity activity;

		Iterator<Activity> it = protocol.iterator();
		while (it.hasNext()) {
			activity = it.next();
			if (activity.getId() == id) {
				if (coveredArea == null)
					coveredArea = activity.getManipulatedArea();
				else
					coveredArea.add(activity.getManipulatedArea());
			}
		}

		return coveredArea;
	}

	// Not currently used:
	// Checks whether {area} is completely within the
	// area manipulated during all activities with {id}
	public boolean isInside(int id, Rectangle2D area) {
		boolean isInside = true;
		float[] coords;
		Point2D p;
		Rectangle2D coveredArea = getCoveredArea(id);

		// If every point on the area's outline is inside coveredArea,
		// the whole area is inside coveredArea
		PathIterator it = coveredArea.getPathIterator(null);

		while (!it.isDone()) {
			it.next();
			coords = new float[2]; // Stores the coordinate of one point
			if (it.currentSegment(coords) == PathIterator.SEG_LINETO) {
				p = new Point2D.Float(coords[0], coords[1]);
				if (!coveredArea.contains(p)) {
					isInside = false;
					break;
				}
				;
			}
		}
		return isInside;
	}

	// Methods for recording image manipulation activities
	// Each of them manipulate a certain area of the image that is to be
	// restored in an Undo operation
	public Rectangle2D recordActivity(int actId, Rectangle rect) {
		Rectangle enlargedRect = rect, trimmedRect = null;
		filename = null;

		if (rect.getWidth() == 0) {
			enlargedRect = new Rectangle((int) rect.getX() - 1, (int) rect
					.getY(), 3, (int) rect.getHeight());
		} else if (rect.getHeight() == 0) {
			enlargedRect = new Rectangle((int) rect.getX(),
					(int) rect.getY() - 1, (int) rect.getWidth(), 3);
		}

		// Fit the area into the image
		Rectangle wholeImage = wizard.wholeImage.getBounds();
		trimmedRect = enlargedRect;
		if (wholeImage.intersects(enlargedRect))
			trimmedRect = wholeImage.intersection(enlargedRect);

		// Store the activity
		guiActions.toggleUndoAction(true);

		filename = "" + protocol.size();
		Activity activity = new Activity(actId, trimmedRect, filename);
		protocol.push(activity);

		// Store the image portion in a recognizable file on harddisk
		if ((actId != Activity.BLACK_PEN) && (actId != Activity.ERASER)
				&& (actId != Activity.FILL) && (actId != Activity.BLACK_LINE)
				&& (actId != Activity.WHITE_LINE)
				&& (actId != Activity.QUADCURVE)
				&& (actId != Activity.BINARIZE)
				&& (actId != Activity.DESPECKLE) && (actId != Activity.SHARPEN)
				&& (actId != Activity.SMOOTH) && (actId != Activity.MINIMUM_OP)
				&& (actId != Activity.BRIGHTNESS)
				&& (actId != Activity.GREYSCALE_CONVERSION)
				&& (actId != Activity.COLOR_QUANTIZATION)
				&& (actId != Activity.GLOBAL_COLORCHANGE)) // Current image
															// saved prior to
															// modification
		{
			portion = imageOperations.copySubImage(
					imagePanel.getCurrentImage(), trimmedRect);
		} else // Modified image restored after modification
		{
			portion = imageOperations.copySubImage(imageBuffer
					.copyFromSnapshot(), trimmedRect);
		}

		// Previous realization caused inconsistencies:
		// fileOperations.saveImage(filename, portion);
		Thread worker = new Thread() {
			public void run() {
				File file = new File(Preferences.protocolPath + File.separator
						+ filename + ".png");
				try {
					OutputStream os = new BufferedOutputStream(
							new FileOutputStream(file));
					ImageIO.write(portion, "PNG", os);
				} catch (Exception e) {
					JOptionPane.showMessageDialog(null,
							"An error occurred while writing the image file.\nError: "
									+ e.getMessage());
				} finally {
				}
			}
		};
		SwingUtilities.invokeLater(worker);

		Wizard.undoHistoryEmptyAfterLastSave = false;

		return trimmedRect;
	}

	// Record tree topology related activities
	public boolean recordActivity(int actId, Object o) {
		// Store the activity
		guiActions.toggleUndoAction(true);

		if (actId == Activity.AS_REFERENCE_NODE) // Stores the reference node's
													// id
		{
			Activity activity = new Activity(actId, o);
			protocol.push(activity);
			return true;
		} else if (actId == Activity.NAME_TAXON) {
			Activity activity = new Activity(actId, o);
			protocol.push(activity);
			return true;
		} else if (actId == Activity.BRANCH_LENGTH) {
			Activity activity = new Activity(actId, o);
			protocol.push(activity);
			return true;
		} else if ((actId == Activity.ADD_INNERNODE)
				|| (actId == Activity.ADD_BRANCH)
				|| (actId == Activity.ADD_TIP)
				|| (actId == Activity.REMOVE_NODE)
				|| (actId == Activity.INSERT_ROOT)
				|| (actId == Activity.REMOVE_BRANCH)
				|| (actId == Activity.REMOVE_SELECTED_NODES)
				|| (actId == Activity.DRAW_BRANCH)
				|| (actId == Activity.FIND_NODES)
				|| (actId == Activity.FIND_BRANCHES)
				|| (actId == Activity.REMOVE_SELECTED_BRANCHES)) {
			Activity activity = new Activity(actId, o);
			protocol.push(activity);

			Wizard.undoHistoryEmptyAfterLastSave = false;

			return true;
		}
		return false;
	}

	public void restorePreviousActivity() {
		if (!protocol.empty()) {
			Activity activity = (Activity) protocol.pop();
			int id = activity.getId();
			Rectangle2D rect = (Rectangle2D) activity.getManipulatedArea();
			String filename = activity.getFilename();

			// Graphical activities are undone here...
			if ((id != Activity.AS_REFERENCE_NODE)
					&& (id != Activity.ADD_INNERNODE)
					&& (id != Activity.ADD_TIP) && (id != Activity.ADD_BRANCH)
					&& (id != Activity.REMOVE_NODE)
					&& (id != Activity.REMOVE_BRANCH)
					&& (id != Activity.REMOVE_SELECTED_NODES)
					&& (id != Activity.REMOVE_SELECTED_BRANCHES)
					&& (id != Activity.DRAW_BRANCH)
					&& (id != Activity.FIND_NODES)
					&& (id != Activity.FIND_BRANCHES)
					&& (id != Activity.NAME_TAXON)
					&& (id != Activity.BRANCH_LENGTH)
					&& (id != Activity.INSERT_ROOT)) {
				// Get image portion in directory /protocol
				File file = new File(Preferences.protocolPath + File.separator
						+ filename + ".png");

				try {
					InputStream is = new BufferedInputStream(
							new FileInputStream(file));
					portion = ImageIO.read(is);
				} catch (IOException e) {
				}

				if (portion != null) {
					imageOperations.copyIntoImage(portion, (int) rect.getX(),
							(int) rect.getY());
				}

				if (protocol.size() == 0)
					guiActions.toggleUndoAction(false);

				file.delete();
			} else // Undo for activities referring to the tree topology
			{
				switch (id) {
				case Activity.AS_REFERENCE_NODE: {

					Integer nodeId = (Integer) activity.getWrappedObject();
					// Check whether the node with {nodeId} is still in the
					// topology
					if (topology.findNode(nodeId)) {
						wizard.referenceNodeId = nodeId;
						topology.redisplayNewickExpression();
					} else // No reference node determined
					{
						wizard.referenceNodeId = -1;
					}
					break;
				}

				case Activity.NAME_TAXON: {
					Vector<?> v = (Vector<?>) activity.getWrappedObject();
					int nodeId = -1;
					String taxonName = null;
					if (v.elementAt(0) instanceof Integer) {
						nodeId = (Integer) v.elementAt(0);
					}
					if (v.elementAt(1) instanceof String) {
						taxonName = (String) v.elementAt(1);
						topology.setTaxonName(nodeId, taxonName);
						imagePanel.updateResultFrame();
					}
					break;
				}

				case Activity.BRANCH_LENGTH: {
					Vector<?> v = (Vector<?>) activity.getWrappedObject();
					int branchId = -1;
					double branchLength = 0.0d;
					if (v.elementAt(0) instanceof Integer) {
						branchId = (Integer) v.elementAt(0);
					}
					if (v.elementAt(1) instanceof Double) {
						branchLength = (Double) v.elementAt(1);
						topology.getBranch(branchId).setUserAssignedLength(
								branchLength);
						// topology.setBranchLength(branchId, branchLength);
						NumberUtility.setNumFractionDigits(topology
								.getBranches());
						imagePanel.updateResultFrame();
					}
					break;
				}

				case Activity.ADD_TIP: {
					restoreTopologyFromMemory(activity);
					break;
				}
				case Activity.ADD_INNERNODE: {
					restoreTopologyFromMemory(activity);
					break;
				}
				case Activity.REMOVE_NODE: {
					restoreTopologyFromMemory(activity);
					break;
				}
				case Activity.INSERT_ROOT: {
					restoreTopologyFromMemory(activity);
					break;
				}

				case Activity.REMOVE_BRANCH: {
					restoreTopologyFromMemory(activity);
					NumberUtility.setNumFractionDigits(topology.getBranches());
					break;
				}
				case Activity.REMOVE_SELECTED_NODES: {
					restoreTopologyFromMemory(activity);
					break;
				}
				case Activity.REMOVE_SELECTED_BRANCHES: {
					restoreTopologyFromMemory(activity);
					NumberUtility.setNumFractionDigits(topology.getBranches());
					break;
				}
				case Activity.DRAW_BRANCH: {
					restoreTopologyFromMemory(activity);
					break;
				}
				case Activity.FIND_NODES: {
					restoreTopologyFromMemory(activity);
					break;
				}
				case Activity.FIND_BRANCHES: {
					restoreTopologyFromMemory(activity);
					wizard.topologyDeterminationFinished = false;
					break;
				}
				}

				if (protocol.size() == 0)
					guiActions.toggleUndoAction(false);
			}

			// Make sure that the program displays a dialog box upon quit to
			// remind the user of saving,
			// if he has modified something after the last save operation
			// occurred
			if (this.protocolEmpty())
				Wizard.undoHistoryEmptyAfterLastSave = true;
			else
				Wizard.undoHistoryEmptyAfterLastSave = false;
		} else {
			guiActions.toggleUndoAction(true);
			System.out
					.println("Internal Error: Protocol is empty, but should not.");
		}
	}

	public void restoreTopologyFromMemory(Activity activity) {
		if (protocol.size() == 0)
			guiActions.toggleUndoAction(false);
		ByteArrayOutputStream b = null;
		ObjectInputStream is = null;
		Object out = activity.getWrappedObject();
		Object o = null;

		if (out instanceof ByteArrayOutputStream) {
			b = (ByteArrayOutputStream) out;
			ByteArrayInputStream in = new ByteArrayInputStream(b.toByteArray());

			try {
				is = new ObjectInputStream(in);
				o = is.readObject();
				if (o instanceof Vector)
					topology.setNodes((Vector<TreeNode>) o);
				o = is.readObject();
				if (o instanceof Vector)
					topology.setBranches((Vector<Branch>) o);
				o = is.readObject();
				if (o instanceof Integer)
					TreeNode.num = (Integer) o;
				o = is.readObject();
				if (o instanceof Integer)
					TreeNode.signalNode = (Integer) o;
				o = is.readObject();
				if (o instanceof Integer)
					Branch.num = (Integer) o;
				o = is.readObject();
				if (o instanceof Integer)
					Tip.nameCnt = (Integer) o;

				is.close();
				b.close();
			} catch (IOException e) {
			} catch (ClassNotFoundException e) {
			}
			;
		}

		NumberUtility.setNumFractionDigits(topology.getBranches());
		topology.transformIntoNewickExpression();
		imagePanel.updateResultFrame();
		imagePanel.revalidate();

	}

	public void reset() {
		// Note: This a runtime protocol; it is not stored along with the image
		protocol = new Stack<Activity>();
		if (guiActions != null)
			guiActions.toggleUndoAction(false);
	}

	// Clears all files in directory /protocol
	public boolean purgeData() {
		if (Preferences.protocolPath != null) {
			try {
				File[] allFiles = new File(Preferences.protocolPath)
						.listFiles();
				for (int i = 0; i < allFiles.length; i++) {
					File file = (File) allFiles[i];
					file.delete();
				}
			} catch (Exception e) {
				System.out.println("Could not delete the protocol files");
				return false;
			}
		}
		return true;
	}

	public boolean protocolEmpty() {
		if (protocol.empty())
			return true;
		else
			return false;
	}
}