## This function returns the _number_ of SNPs associated with GO terms of the
## different levels of the gotermsByLevel list, which contains the GO terms 
## associated with the different levels of the MF, BP and CC ontologies. The
## gotermsByLevel list was generated by the 'ComputeLevels' function. .

## The gotermsByLevel parameter is either ontology.levels[["MF"]],
## ontology.levels[["BP"]], or ontolog.levels[["CC"]].

ComputeSNPsByLevel <-
function(gotermsByLevel, gotermMappings, numInformativeSNPs, ontoName, verbose) {
  snpLength <- list()
  snpLength[["candidates"]] <- vector("integer", length(gotermsByLevel))
  snpLength[["noncandidates"]] <- vector("integer", length(gotermsByLevel))

  ## Iterate through the ontology.levels from bottom to top
  ## Level i contains all the SNPs associated to level(i+1) plus 
  ## the SNPs associated to level(i).
  for(type in c("candidates", "noncandidates")) {
    if(verbose) {
      message(
        paste(type, "associated with the", ontoName, "ontology:")
      )
    }
    level <- list()
    ## Instead of letting the level[[type]] vector grow by appending new
    ## values like level[[type]] <- c(level[[type]], newValues), we pre-
    ## allocate space and increase the vector size if needed.
    ## See Chapter 2 of Patrick Burns' "The R Inferno"
    if(numInformativeSNPs[[type]] == 0){
      vlen = 5
    } else {
      vlen = numInformativeSNPs[[type]]
    }

    level[[type]] <- vector("integer",vlen)

    ## upper is the number of currently stored elements
    upper <- 0
    for(i in length(gotermsByLevel):1) {
      if(i == length(gotermsByLevel)) {
        goterms <- gotermsByLevel[[i]]
      }
      else{
        goterms <- 
          gotermsByLevel[[i]][
            (gotermsByLevel[[i]] %in% gotermsByLevel[[i+1]]) == FALSE
          ]
      }
    
    
      for(goterm in goterms) {
        regions <- gotermMappings[[goterm]]
       
        if(length(regions[[type]]) > 0) {
          lower <- upper + 1
          upper <- lower + length(regions[[type]]) - 1
        
          if(upper > length(level[[type]])) {
            tmp <- level[[type]]
            level[[type]] <- vector("integer", upper * 2)
            level[[type]][1:length(tmp)] <- tmp
          }
      
          level[[type]][lower:upper] <- regions[[type]]
        }
      }

      if(upper > 0) {
        unq <- unique(level[[type]][1:upper])      
        upper <- length(unq)
        level[[type]][1:upper] <- unq
      }

      snpLength[[type]][i] <- upper
      if(verbose) {
        message(
          paste("Level ", i, ": ", snpLength[[type]][i], sep='')
        )
      }

    }
  }

  ## Return a list
  list(
    "LengthCandidatesByLevel"=snpLength[["candidates"]], 
    "LengthNoncandidatesByLevel"=snpLength[["noncandidates"]]
  )
}
