/*
 * main.cpp
 *
 *  Created on: Mar 31, 2011
 *      Author: philipp_
 */

#include <iostream>
#include <cstdlib>
#include <map>
#include <dlfcn.h>
#include <string.h>
#include <cmath>

#include "IAlignment.h"

#include "SimpleLog.h"
#include "SimpleConfig.h"

enum AligmentMode {
	LOCAL = 0, SEMIGLOBAL = 1
};

enum AlignFormat {
	TEXT = 0, CIGAR = 0x100
};

#define pRef pBuffer1
#define pRead pBuffer2

#define pCigar pBuffer1
#define pMD pBuffer2

std::map<int, void *> loadedDLLs;

SimpleConfig config;
SimpleLog slog;

using std::cout;
using std::endl;
using std::cerr;


void * GetDLLFunc(int const dll, char const * const name, bool required = true) {
	void * sdll = loadedDLLs[dll];
	void * pf = dlsym(sdll, name);
	if (required && (pf == 0)) {
		slog.Error("Unable to load function %s", name);
	}
	return pf;
}

int const InitDLL(char const * const filename) {
	try {
		void * sdll = dlopen(filename, RTLD_LAZY);

		if (!sdll) {
			slog.Error("Cannot open library: %s", dlerror());
			return -1;
		}

		int pos = loadedDLLs.size();
		loadedDLLs[pos] = sdll;
		pfSetLog SetLog = (pfSetLog) GetDLLFunc(pos, "SetLog");
		if (SetLog != 0) {
			SetLog(&slog);
		}

		pfSetConfig SetConfig = (pfSetConfig) GetDLLFunc(pos, "SetConfig");
		if (SetConfig != 0) {
			SetConfig(&config);
		}

		return pos;
	} catch (int err) {
		slog.Error("Unable to load DLL %s (Error %i)", filename, err);
	}
	return 0;
}

size_t pad(char const * * sequences, int const n) {
	size_t maxLen = 0;
	for (int i = 0; i < n; ++i) {
		maxLen = std::max(maxLen, strlen(sequences[i]));
	}
	for (int i = 0; i < n; ++i) {
		char * tmp = new char[maxLen];
		size_t len = strlen(sequences[i]);
		for (size_t j = 0; j < maxLen; ++j) {
			tmp[j] = (j < len) ? sequences[i][j] : '\0';
		}
		//delete[] sequences[i];
		sequences[i] = tmp;
	}
	return maxLen;
}

int main(int argc, char *argv[]) {

	char const
			* reads[] =
					{
							"CACACCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCT",
							"CACACCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCT",
							"CACACCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCT",
							"CACACCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCT",
							"CACACCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCT",
							"GGGTAAGTTGAGAGACAGGTTGGACAGGGTTAGATTAGGGCTGTGTTAGGGTAGTGTTAGGATGTGTGTGTGTGGGTGTGGTGTGGTGTGTGGTGTGGTG",
							"GGGTAAGTTGAGAGACAGGTTGGACAGGGTTAGATTAGGGCTGTGTTAGGGTAGTGTTAGGATGTGTGTGTGTGGGTGTGGTGTGGTGTGTGGTGTGGTG",
							"GGGTAAGTTGAGAGACAGGTTGGACAGGGTTAGATTAGGGCTGTGTTAGGGTAGTGTTAGGATGTGTGTGTGTGGGTGTGGTGTGGTGTGTGGTGTGGTG" };
	char const
			* refs[] =
					{
							"GTTGGGTGACACACCCACACACCACACCACACACCAGACCCACACCCACAAACACACATCCTAAGACTGCCCTAAAACTGCCCTAATCTAACCCTGGCCAACCTGTCTCTGTGGTCA",
							"TCAACTTCCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCTCCCTCCAT",
							"AGGGTAACGCACACCCACACACCACACCACACACCAGACCCACACCCACACACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCTTTGGGTG",
							"TTGGAGGGCACACCCACACACCACACCACACACCAGACCCACACCCACACACAAAACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCTTAACTTTG",
							"CCCTCCATTACACACCCACACACCACACCACACACCAGACCCACACCCAAGGACACACATCCTAAGACTGCCCTAAAACAGCCCTAATCTAACCCTGGCCAACCTGTCTCTCCCTGC",
							"TAATTGGAGGGTAAGTTGAGAGACAGGTTGGACAGGGTTAGATTAGGGCTGTGTTAGGGTAGTGTTAGGATGTGTGTGTGTGGGTGTGGTGTGGTGTGTGGTGTGGGTAACG",
							"TCCATTAAAGTTGAGAGACAGGTTGGACAGGGTTAGATTAGGGCTGTGTTAGGGTAGTGTTAGGATGTGTGTGTGTGGGTGTGGTGTGGTGTGTGGTGTGGTGCCCTGCCTG",
							"TATTACCCTGGGGTAAGTTGAGAGACAGGTTGGACAGGGTTAGATTAGGGCTGTGTTAGGGTAGTGTTAGGATGTGTGTGTGTGGGTGTGGTGTGGTGTGTGGTGTGGTGCCTCCA" };
	int seqNumber = 8;
	size_t maxReadLen = pad(reads, seqNumber);
	size_t maxRefLen =	pad(refs, seqNumber);

	cout << "MaxLen: " << maxReadLen << ", " << maxRefLen << endl;

	try {

		char const * libPath = "./libMASonOpenClGpu.so";

		int const dll = InitDLL(libPath);
		pfCreateAlignment createAlignment = (pfCreateAlignment) GetDLLFunc(dll,
				"CreateAlignment");
		pfDeleteAlignment delAlignment = (pfDeleteAlignment) GetDLLFunc(dll,
				"DeleteAlignment");
		IAlignment * mason = 0;


		int reportType = TEXT;
		int devId = 0;

		config.read_size = maxReadLen;
		//If corridor length is change, the length of the reference sequences has to be changed accordingly.
		config.corridor = 10;
		//Number of CPU cores used (0 = all).
		config.cores = 0;

		mason = createAlignment(devId | reportType);

		//Score computation
		float * scores = new float[seqNumber];
		int batch_size = mason->GetScoreBatchSize();
		cout << "Batchsize for computing scores: " << batch_size << std::endl;

		mason->BatchScore(LOCAL, seqNumber, &refs[0], &reads[0],
				&scores[0], 0);

		cout << "Scores: " << std::endl;
		for (int i = 0; i < seqNumber - 1; ++i) {
			cout << scores[i] << ", ";
		}
		cout << scores[seqNumber - 1] << std::endl;

		//Alignment computation
		Align * alignments = new Align[seqNumber];
		for (int i = 0; i < seqNumber; ++i) {
			alignments[i].pRef = new char[maxReadLen * 2];
			alignments[i].pRead = new char[maxReadLen * 2];
		}

		mason->BatchAlign(LOCAL, seqNumber, &refs[0], &reads[0],
				&alignments[0], 0);
		cout << "Alignments: " << std::endl;
		for (int i = 0; i < seqNumber; ++i) {
			cout << "Ref:   " << alignments[i].pRef << std::endl;
			cout << "Reads: " << alignments[i].pRead << std::endl << std::endl;
		}

		delAlignment(mason);

	} catch (char const * ex) {
		cout << "Exception: " << ex << endl;
	} catch (char const* ex) {
		cout << "Exception: " << ex << endl;
	}
	return 0;
}
