/***************************************************************************
 *   Copyright (C) 2010 by Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler   *
 *   minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef TREE_H
#define TREE_H

/**
describes a tree

	@author Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler 
	<minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at>
*/

#include "node.h"
#include "split.h"
#include "utility.h"

using namespace std;

class Tree{
public:
	/**
		Default constructor
	*/
    Tree();
	/**
		Construct the tree from a treeFile
		@param treeFile name of the file containing the tree
	*/
	Tree(const char *treeFile);
	/**
		Copy constructor
		@param atree another tree
	*/
	Tree(Tree &atree);	

	/**
		virtual destructor
	*/
	virtual ~Tree();

	/**====================================================*/
	/**======= READ TREE FROM FILE IN NEWICK FORMAT =======*/
	/**====================================================*/
	/**
		read the tree from an input file
		@param infile the input file name
	*/
    void readTree(const char *infile);

	/**
		read the tree from an input stream
		@param in the input stream
	*/
    void readTree(ifstream &in);

	/**
		parse the tree (node) from input stream
		@param in the input stream
		@param ch (IN/OUT) current char
		@param currentNode (IN/OUT) the current node
		@param branchLen (OUT) branch length associated to the current node
	*/	
	void parseFile(ifstream &in, char &ch, Node* &currentNode, double &branchLen);

	/**
		initialize tree, set node structure (nodeID)
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
	void initializeTree(Node *node = NULL, Node *dad = NULL);

	/**
		release the memory
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
    void freeNode(Node *node = NULL, Node *dad = NULL);

	/**--------------------------------------------------------*/
	/**----- PRINT OUT TREE WITH DIFFERENT INFORMATION --------*/
	/**--------------------------------------------------------*/
	/**
		print the tree to the output file in newick format
		@param outfile the output file.
		@param brtype type of branch to print
	*/
	void printTree(const char *outfile, BranchType brtype = BR_LEN);

	/**
		print the tree to the output file in newick format
		@param out the output stream.
		@param brtype type of branch to print
	*/
	void printTree(ostream &out, BranchType brtype = BR_LEN);

	/**
		print the tree to the output file in newick format
		@param out the output file.
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
		@param brtype type of branch to print
	*/
	void printTree(ostream &out, BranchType brtype, Node *node, Node *dad = NULL);

	/**
		print information
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
	void printInfo(Node *node = NULL, Node *dad = NULL);

	/********************************************************
	DRAW TREE
	********************************************************/

	void drawTree(ostream &out, int brtype = WT_BR_SCALE + WT_INT_NODE);
	void drawTreeBrId(ostream &out);

	void drawTree(ostream &out, int brtype, double brscale, IntVec &sub_tree_br, 
		Node *node = NULL, Node *dad = NULL);
	void drawTree2(ostream &out, int brtype, double brscale, IntVec &sub_tree_br, 
		Node *node = NULL, Node *dad = NULL);
	void drawTree3(ostream &out, double brscale, IntVec &sub_tree_br, 
		Node *node = NULL, Node *dad = NULL);
	
	/**-------------------------------------------------------*/
	/**----- GET INFORMATION----------------------------------*/
	/**-------------------------------------------------------*/
	string getRootName(){
		return root->name;
	}

	/**--------------------------------------------------------*/
	/**---- SEARCHING AND BASIC OPERATIONS ON TREE ------------*/
	/**--------------------------------------------------------*/
	/**
		find a node with a corresponding name
		@param name node name
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
		@return node if found, otherwise NULL
	*/
	Node *findNodeName(string &name, Node *node = NULL, Node* dad = NULL);

	/**
		find seqID of a node with a given name
		@param name node name
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
		@return seqID if found, otherwise -1
	*/
	int findSeqID(string &name, Node *node = NULL, Node* dad = NULL);

	/**
		find a branch/split on the tree corresponding to the given split
		@param querySplit the given split to find
		@param resp the split at the current node (directed by dad)
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
		@return a node if found, other NULL
	*/
	Node* findSplit(const Split querySplit, Split *resp, Node *&retNode2, Node* node = NULL, Node *dad = NULL);

	/**
		@return maximum path length from root node to taxa
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
	double treeDepth(Node *node = NULL, Node *dad = NULL);

	/**
		update root of the tree
		@param name of the outgroup to root the tree		
	*/
	void updateRoot(string outgroupName="");

	/**
		update root of the tree on the branch seperating the given taxon set
		@param taxa a set of taxon's names
	*/
	void updateRoot(StringVec taxa);
	
	/**
		update seqID
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
	void updateSeqID(StringVec leafName, Node *node = NULL, Node *dad = NULL);

	/**
		set Id for the branches of a subtree rooted at node
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
	void setBrId(Node *node = NULL, Node *dad = NULL);

	/**
		reset brID for the whole tree
	*/
	void resetBrId();

	/**
		convert a (sub) tree rooted at node into a set of splits and add them into a map between brID and split. Tree is treated as rooted tree. In the splits, taxon order is the same as in the alignment.
		@param brSplit (OUT) returned map of brIDs and splits
		@param resp (OUT) the split at current node
		@param node starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
	*/
	void convertSplits(MapBrSplit &brSplit, Split *resp, Node* node = NULL, Node *dad = NULL);

	/**----------------------------------------*/
	/**------CHECKING THE TREE-----------------*/
	/**----------------------------------------*/	
	/**
		check tree is bifurcating tree (every leaf with level 1 or 3)
		@param node the starting node, NULL to start from the root
		@param dad dad of the node, used to direct the search
		@param stop (IN/OUT) set = true to stop the search
		@NOTE tree is bifurcating if stop == false at the end
	*/
	void checkValidTree(bool& stop, Node *node = NULL, Node *dad = NULL);

	/*-------------ATTRIBUTES---------------*/
	/*--------------------------------------*/
	/**
		Root of the tree. When reading from file, it's an artificial root. If user sets it, it will be updated (will change) by this time.
	*/
	Node* root;
	/**
		Number of leaves on the tree
	*/
	int leafNum;
	/**
		Total number of nodes on the tree
	*/
	int nodeNum;
	/**
		Total number of branches on the tree
	*/
	int brNum;
	/**
		if the tree is really rooted infered from tree-file. TRUE if
		for example: (1) (newick format):_num; //_num>0
					(2) (newick format); //and the artificial root has exactly 2 neighbor --> create a new Node, call _root, _root.neighbor=root, assign root=_root. This is to make sure that root is a leaf.
		TODO: after parseFile (node, branchLength)
		if (branchLength>0): create a new Node, call _root, _root.neighbor=root, assign root=_root.
		else if (root has 2 neighbor): create a new Node, call _root, _root.neighbor=root, assign root=_root.
		==> rewrite the code for more efficient
	*/
	bool rooted;
	Node* outgroup;
};

#endif
