/***************************************************************************
 *   Copyright (C) 2010 by Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler   *
 *   minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "tree.h"

Tree::Tree(){
	root = NULL;
	leafNum = 0;
	nodeNum = 0;
	brNum = 0;
	rooted = false;
	outgroup = NULL;
}

Tree::Tree(const char *treeFile){
	readTree(treeFile);	
}

Tree::Tree(Tree &atree){
	root = atree.root;
	leafNum = atree.leafNum;
	nodeNum = atree.nodeNum;
	brNum = atree.brNum;
	rooted = atree.rooted;
	outgroup = atree.outgroup;
	//have to delete the root when exchange to another object 
 	atree.root = NULL;
 	atree.outgroup = NULL;
}

Tree::~Tree(){
	if (root != NULL)
		freeNode();
	root = NULL;
	outgroup = NULL;
}

/*!
    \fn Tree::freeNode(Node *node = NULL, Node *dad = NULL)
 */
void Tree::freeNode(Node *node, Node *dad)
{
    if (!node) node = root;
	NeighborVec::reverse_iterator it;
	for (it = node->neighbors.rbegin(); it != node->neighbors.rend(); it++)
		if ((*it)->node != dad)
		{			
			freeNode((*it)->node, node);
		}
	delete node;
}

/*!
    \fn Tree::readTree(const char *infile)
 */
void Tree::readTree(const char *infile)
{
	cout << "\nReading tree file " << infile << " ..." << endl;
	ifstream in;
	try {
		in.exceptions(ios::failbit | ios::badbit);
		in.open(infile);
		readTree(in);
		in.close();
	} catch (ios::failure) {
		outError(ERR_READ_INPUT, infile);		
	}

// 	cout << "Tree contains " << leafNum - rooted << 
// 		" taxa and " << nodeNum-1-rooted << " branches" << endl;
	cout << "Tree contains " << leafNum << 
		" taxa and " << nodeNum-1 << " branches" << endl;
}
/*!
    \fn Tree::readTree(ifstream &in)
 */
void Tree::readTree(ifstream &in)
{
	try {
		char ch;
		in >> ch;
		if (ch != '(')
			throw "Tree file not started with an opening-bracket '('";
	
		leafNum = 0;
		rooted=false;
	
		double branch_len;
		Node *node;
		parseFile(in, ch, node, branch_len);
		nodeNum = leafNum;	
		//the tree is rooted (inferred from input file)
		if (node->neighbors.size() == 2 || branch_len > 0.0) {
			if (branch_len == -1.0) branch_len = 0.0;		
			if (branch_len < 0.0) 
				throw ERR_NEG_BRANCH; 
			rooted = true;			
			root = new Node(nodeNum, ROOT_NAME);
			nodeNum++;
 			root->neighbors.push_back(new Neighbor(node, branch_len));
 			node->neighbors.push_back(new Neighbor(root, branch_len));			
			//leafNum++;	//COMMENT ON 2009.09.13
		}		
		// make sure that root is a leaf
// 		assert(root->isLeaf());	
		if (in.eof() || ch != ';')
			throw "Tree file must be ended with a semi-colon ';'";
	} catch (bad_alloc) {
		outError(ERR_NO_MEMORY);
	} catch (const char *str) {
		outError(str);
	} catch (char *str) {
		outError(str);
	} catch (string str) {
		outError(str);
	} catch (ios::failure) {
		outError(ERR_READ_INPUT);
	} catch (...) {
		// anything else
		outError(ERR_READ_ANY);
	}		 		
	initializeTree();
	resetBrId();
	//bool stop = false;
	//checkValidTree(stop);
}

/*!
    \fn Tree::parseFile(ifstream &in, char &ch, Node* &currentNode, double &branchLen)
 */
void Tree::parseFile(ifstream &infile, char &ch, Node* &currentNode, double &branchLen)
{
	Node *node;
	int maxlen = 100;
	char seqname[100];
	int seqlen;
	double brlen;
	branchLen = -1.0;

	currentNode = new Node();

	if (ch == '(') { // internal node		
		infile >> ch;
		while (ch != ')' && !infile.eof())
		{
			node = NULL;
			parseFile(infile, ch, node, brlen);			
 			currentNode->neighbors.push_back(new Neighbor(node, brlen));
 			node->neighbors.push_back(new Neighbor(currentNode, brlen));			
			if (infile.eof())
				throw "Expecting ')', but end of file instead";
			if (ch == ',')
				infile >> ch;
			else if (ch != ')') {
				string err = "Expecting ')', but found '";
				err += ch;
				err += "' instead";
				throw err;
			}
		}
		if (!infile.eof()) infile >> ch;
	}
	// now read the node name
	seqlen = 0;
	while (!is_newick_token(ch) && !controlchar(ch) && !infile.eof() && seqlen < maxlen)
	{
		seqname[seqlen] = ch;
		seqlen++;
		ch = infile.get();
	}
	if (controlchar(ch) && !infile.eof()) 
		infile >> ch;
	if (seqlen == maxlen)
		throw "Too long name ( > 100)";
	seqname[seqlen] = 0;
	if (seqlen == 0 && currentNode->isLeaf())
		throw "A taxon has no name.";
	if (seqlen > 0) 
		currentNode->name += seqname;
	if (currentNode->isLeaf()) {
		// is a leaf, assign its ID
		currentNode->id = leafNum;
		currentNode->seqID = leafNum;
 		if (leafNum == 0)
 			root = currentNode;
		leafNum++;
	}

	if (ch == ';' || infile.eof())
		return;
	if (ch == ':')
	{
		infile >> ch;
		seqlen = 0;
		while (!is_newick_token(ch) && !controlchar(ch) && !infile.eof() && seqlen < maxlen)
		{
			seqname[seqlen] = ch;
			seqlen++;
			ch = infile.get();
		}
		if (controlchar(ch) && !infile.eof()) 
			infile >> ch;
		if (seqlen == maxlen || infile.eof())
			throw "branch length format error.";
		seqname[seqlen] = 0;
		branchLen = convert_double(seqname);
	}
}

/*!
    \fn Tree::initializeTree(Node *node = NULL, Node *dad = NULL)
 */
void Tree::initializeTree(Node *node, Node *dad)
{	
	if (!node)
		node = root;		
	if (!node->isLeaf())
	{
		node->id = nodeNum;
 		nodeNum++;
		//node->name = node->id;
	}	

 	FOR_NEIGHBOR_IT(node, dad, it)			
 		initializeTree((*it)->node, node);		
}

/**=============================================================*/
/**                    PRINT OUT                                */
/**=============================================================*/
void Tree::printTree(const char *ofile, BranchType brtype)
{
	try {
		ofstream out;
		out.exceptions(ios::failbit | ios::badbit);
		out.open(ofile);
		printTree(out, brtype);
		out.close();
		cout << "Tree was printed to " << ofile << endl;
	} catch (ios::failure) {
		outError(ERR_WRITE_OUTPUT, ofile);
	}
}

void Tree::printTree(ostream &out, BranchType brtype) {
	if (root->isLeaf()) {
		if (root->neighbors[0]->node->isLeaf()) {
			// tree has only 2 taxa!
			out << "(";
			printTree(out, brtype, root);
			out << "," << root->neighbors[0]->node->name;
			switch (brtype) {
			case BR_LEN: out << ":0"; break;
			case BR_ID: out << ":" << root->neighbors[0]->brId; break;
			default: break;
			}
			out << ")";
		} else
			// tree has more than 2 taxa
			printTree(out, brtype, root->neighbors[0]->node);
	} else 
		printTree(out, brtype, root);

	out << ";";
}

void Tree::printTree(ostream &out, BranchType brtype, Node *node, Node *dad)
{
// 	out.setf(ios::oct,ios::fixed);
	out.precision(10);
	out << std::fixed;
	if (!node) node = root;
	if (node->isLeaf()) {
		out << node->name;
		switch (brtype) {
		case BR_LEN: out << ":" << node->neighbors[0]->length; break;
		case BR_ID: out << ":" << node->neighbors[0]->brId; break;
		default: break;
		}
	} else {
		// internal node
		out << "(";
		bool first = true;
		double length = 0.0;
		int branchId = -1;		
		FOR_NEIGHBOR_IT(node, dad, it) {
 			if ((*it)->node->name != ROOT_NAME) {
				if (!first)
					out << ",";
				printTree(out, brtype, (*it)->node, node);
				first = false;
			} else {			
				length = (*it)->length;
				branchId = (*it)->brId;
			}
		} else {
			length = (*it)->length;
			branchId = (*it)->brId;
		}
		out << ")";
		if (!node->name.empty())
			out << node->name;
		if (dad != NULL || length > 0.0 || branchId != -1){
			switch (brtype) {
			case BR_LEN: out << ":" << length; break;
			case BR_ID: out << ":" << branchId; break;	
			default: break;
			}
		}
	}
}

void Tree::printInfo(Node *node, Node *dad)
{
	cout.precision(10);
	cout << std::fixed;
 	if (node == NULL) node = root;
	FOR_NEIGHBOR_IT(node, dad, it) {		
		cout << node->name << "_" << node->id << "_" << node->seqID << "\t" << (*it)->node->name << "_" << (*it)->node->id << "_" << (*it)->node->seqID << " : " << "\t" << (*it)->length << "_" << (*it)->brId << endl;
		printInfo((*it)->node, node);
	}
}

void Tree::drawTreeBrId(ostream &out){
	IntVec sub_tree_br;
	Node *node = root;
	/*Node *dad;
	if (node->isLeaf()){
		dad = node;
		node = node->neighbors[0]->node;
	}
	drawTree3(out,scale,sub_tree_br,node,dad); */
	if (node->isLeaf()) node = node->neighbors[0]->node;
	double scale = 60.0/treeDepth(node);
	drawTree3(out,scale,sub_tree_br);
}

void Tree::drawTree(ostream &out, int brtype) {
	IntVec sub_tree_br;	
	Node *node = root;
	if (node->isLeaf()) node = node->neighbors[0]->node;
	double scale = 60.0/treeDepth(node);
//	if (verbose_mode >= VB_DEBUG)
//	cout << "Tree depth: " << scale<< endl;
	if (brtype & WT_INT_NODE){
		out << "drawTree2 is called." << endl;
		drawTree2(out, brtype, scale, sub_tree_br);
	}
	else{
		out << "drawTree is called. " << endl;
		drawTree(out, brtype, scale, sub_tree_br);
	}	
	out << endl;
}

void Tree::drawTree(ostream &out, int brtype, double brscale, IntVec &subtree_br, Node *node, Node *dad) {
	int i, br_len = 3;
	if (!node) {
		node = root;
		if (node->isLeaf()) node = node->neighbors[0]->node;
	} else {
		if (brtype & WT_BR_SCALE) {
			br_len = floor(node->findNeighbor(dad)->length * brscale)-1;
			if (br_len < 3) br_len = 3;
			//if (!node->isLeaf() && br_len < 4) br_len = 4;
		}
		out << '+';
		if ((brtype & WT_INT_NODE) && !node->isLeaf()) {
			string str = convertIntToString(node->id);
			for (i = 0; i < br_len-str.length(); i++) out << '-';
			out << node->id;
		} else 
		for (i = 0; i < br_len; i++) out << '-';
	} 
	if (node->isLeaf()) {
		out << node->name; 
		if (brtype & WT_TAXON_ID)
			out << " (" << node->id << ")";
		out << endl;
		return;
	}
	int descendant_cnt = node->degree();
	if (dad) descendant_cnt--;
	int cnt = 0;
	subtree_br.push_back(br_len);
	FOR_NEIGHBOR_IT(node, dad, it) {
		if (cnt == descendant_cnt-1) 
			subtree_br.back() = -subtree_br.back();
		
		drawTree(out, brtype, brscale, subtree_br, (*it)->node, node);
		cnt++;
		if (cnt == descendant_cnt) break;
		for (IntVec::iterator it = subtree_br.begin()+1; it != subtree_br.end(); it++) 
		{
			if ((*(it-1)) > 0) out << '|'; else out << ' ';
			for (i = 0; i < abs(*it); i++) out << ' ';
		}
	}
	subtree_br.pop_back();
}

void Tree::drawTree2(ostream &out, int brtype, double brscale, IntVec &subtree_br, Node *node, Node *dad) {
	int i, br_len = 3;
	IntVec::iterator ii;
	if (!node) {
		node = root;
		if (node->isLeaf()) node = node->neighbors[0]->node;
	} else {
		if (brtype & WT_BR_SCALE) {
			br_len = floor(node->findNeighbor(dad)->length * brscale)-1;
			if (br_len < 3) br_len = 3;
		}
	} 
	if (node->isLeaf()) {
		for (ii = subtree_br.begin()+1; ii != subtree_br.end(); ii++) {
			if (abs(*(ii-1)) > 1000) out << ' '; else out << '|';
			int num = abs(*ii);
			if (num > 1000) num -= 1000;
			for (i = 0; i < num; i++) out << ' ';
		}
		out << '+';
		for (i = 0; i < br_len; i++) 
			out << '-';
		out << node->name; 
		if (brtype & WT_TAXON_ID)
			out << " (" << node->id << ")";
		if (brtype & WT_BR_LEN)
			out << " " << node->neighbors[0]->length;		
		//out << " ";
		//copy (subtree_br.begin(), subtree_br.end(), ostream_iterator<int> (out, " "));
		out << endl;
		return;
	}
	int descendant_cnt = node->degree();
	if (dad) descendant_cnt--;
	int cnt = 0;
	bool first = true;

	br_len = br_len+1000;
	FOR_NEIGHBOR_IT(node, dad, it) {
		if (cnt == descendant_cnt-1)
			br_len = -br_len;
		subtree_br.push_back(br_len);

		drawTree2(out, brtype, brscale, subtree_br, (*it)->node, node);
		subtree_br.pop_back();
		if (br_len > 1000) br_len -= 1000;
		cnt++;
		if (cnt == descendant_cnt) break;
		if (subtree_br.size() > 1)
		for (ii = subtree_br.begin()+1; ii != subtree_br.end(); ii++) {
			if (abs(*(ii-1)) > 1000) out << ' '; else out << '|';
			if (ii == subtree_br.begin()) continue;
			int num = abs(*ii);
			if (num > 1000) num -= 1000;
			for (i = 0; i < num; i++) out << ' ';
		} 
		if (first) {
			if (dad) {
				out << '+';
				for (i = 0; i < abs(br_len); i++) 
					out << '-';
			}
			out << node->id;			
			if (!node->name.empty())
				out << " " << node->name;				
			if (brtype & WT_BR_LEN && dad)
				out << " " << node->findNeighbor(dad)->length;			
			if (!subtree_br.empty())
			if (subtree_br.back() >1000) 
				subtree_br.back() -= 1000;
			else if (subtree_br.back() < 0) 
				subtree_br.back() -= 1000;
		} else {
			if (dad) {
				if (abs(subtree_br.back()) > 1000) out << ' '; else out << "|";
				for (i = 0; i < abs(br_len); i++) 
					out << ' ';
			}
			out << "|";
		} 
		//out << " ";
		//copy (subtree_br.begin(), subtree_br.end(), ostream_iterator<int> (out, " "));
		out << endl;
		first = false;
	}
}

void Tree::drawTree3(ostream &out, double brscale, IntVec &subtree_br, Node *node, Node *dad) {
	int i, br_len = 3;
	IntVec::iterator ii;
	if (!node) {
		node = root;
// 		if (node->isLeaf()) node = node->neighbors[0]->node;
		if (node->isLeaf()){
			if ( strcmp(node->name.c_str(),ROOT_NAME) == 0 ){
				dad = node;
				node = node->neighbors[0]->node;
			}
			else node = node->neighbors[0]->node;
		}
	} else {		
			br_len = floor(node->findNeighbor(dad)->length * brscale)-1;
			if (br_len < 3) br_len = 3;
	} 
	if (node->isLeaf()) {
		for (ii = subtree_br.begin()+1; ii != subtree_br.end(); ii++) {
			if (abs(*(ii-1)) > 1000) out << ' '; else out << '|';
			int num = abs(*ii);
			if (num > 1000) num -= 1000;
			for (i = 0; i < num; i++) out << ' ';
		}
		out << '+';
		for (i = 0; i < br_len; i++) 
			out << '-';
		out << node->name; 
		out << " (" << node->neighbors[0]->brId << ")";
		out << endl;
		return;
	}
	int descendant_cnt = node->degree();
	if (dad) descendant_cnt--;
	int cnt = 0;
	bool first = true;
	if (dad == root )// && strcmp(dad->name.c_str(),ROOT_NAME) == 0)
		first = false;

	br_len = br_len+1000;
	FOR_NEIGHBOR_IT(node, dad, it) {
		if (cnt == descendant_cnt-1)
			br_len = -br_len;
		subtree_br.push_back(br_len);

		drawTree3(out, brscale, subtree_br, (*it)->node, node);
		subtree_br.pop_back();
		if (br_len > 1000) br_len -= 1000;
		cnt++;
		if (cnt == descendant_cnt) break;
		if (subtree_br.size() > 1)
		for (ii = subtree_br.begin()+1; ii != subtree_br.end(); ii++) {
			if (abs(*(ii-1)) > 1000) out << ' '; else out << '|';
			if (ii == subtree_br.begin()) continue;
			int num = abs(*ii);
			if (num > 1000) num -= 1000;
			for (i = 0; i < num; i++) out << ' ';
		} 
		if (first) {
			if (dad) {
				out << '+';
				for (i = 0; i < abs(br_len); i++) 
					out << '-';
 			/*	if (!node->name.empty())
				out << node->name << " ";*/
//  			if (dad)
				out << "(" << node->findNeighbor(dad)->brId << ")";
			}
			else
				out << "|";		
			if (!subtree_br.empty())
			if (subtree_br.back() >1000) 
				subtree_br.back() -= 1000;
			else if (subtree_br.back() < 0) 
				subtree_br.back() -= 1000;
		} else {
			if (dad) {
				if (dad == root)
					//out << "ROOT";// << " (" << dad->neighbors[0]->brId << ")";
					out << "ROOT (" << dad->neighbors[0]->brId << ")";
				else{
					if (abs(subtree_br.back()) > 1000) out << ' '; else out << "|";
					for (i = 0; i < abs(br_len); i++) 
						out << ' ';
 					out << "|";
				}
			}
 			if (dad != root) out << "|";
		} 		
		out << endl;
		first = false;
	}
}

/**===============================================================================*/
/**            SEARCHING AND BASIC OPERATIONS ON TREE                             */
/**===============================================================================*/
Node *Tree::findNodeName(string &name, Node *node, Node *dad) {
	if (!node) node = root;
	if (node->name == name) return node;
	FOR_NEIGHBOR_IT(node, dad, it) {
		Node *res = findNodeName(name, (*it)->node, node);
		if (res) return res;
	}
	return NULL;
}

int Tree::findSeqID(string &name, Node *node, Node *dad) {	
	Node *res = findNodeName(name,node,dad);
	if (res) 
	{		
		return res->seqID;
	}
	return -1;
}

void Tree::checkValidTree(bool &stop, Node *node, Node *dad)
{
	if (!node) node = root;
	if (node->neighbors.size() != 1 && node->neighbors.size() != 3) {
		cout << "Tree is not bifurcating." << endl;
		stop = true;
		return;
	}
	//for (int i = 0; i < node->neighbors.size(); i++)
		//if (node->neighbors[i]->node != dad) {
	FOR_NEIGHBOR_IT(node, dad, it) {
		checkValidTree(stop, (*it)->node, node);
		if (stop) 
			return;
	}
}

double Tree::treeDepth(Node *node, Node *dad)
{
	if (!node) node = root;
	double maxsum = 0.0;
	FOR_NEIGHBOR_IT(node, dad, it) {
		double len = (*it)->length;
		if (len < 0.0) len = 0.0;
		double sum = len + treeDepth((*it)->node, node);
		if (sum > maxsum) maxsum = sum;
	}
	return maxsum;
}

void Tree::updateRoot(string outgroupName)
{
	if (rooted == true && outgroupName != "")
		outError(ERR_CONFLICT_ROOT);
	if (strcmp(root->name.c_str(),ROOT_NAME) == 0 )
		outError(ERR_CONFLICT_ROOT);
	if (!rooted)
	{
		//user provides an outgroup, find node with outgroupName in the tree
		if (outgroupName != "")
		{
			Node *newOutGroup = findNodeName(outgroupName);
			if (newOutGroup == NULL)
			{
				string err = ERR_NO_LEAF_NAME;
				err += outgroupName;
				outError(err);
			}
			if (!newOutGroup->isLeaf())
			{
				string err = ERR_ROOT_AT_INNER;
				err += outgroupName;
				outError(err);
			}
			//found Node with outgroupName ==> assign this Node to outgroup
			outgroup = newOutGroup;
		}
		else
			//user does not provide any outgroup, assign outgroup to the first leaf in Nexus file by default
			outgroup = root;
		
		//create an internal node as the root of the binary tree, determined by outgroup
		Node* newRoot = new Node(nodeNum, INNER_ROOT_NAME);
		double branch_len = outgroup->neighbors[0]->length;
		newRoot->neighbors.push_back(new Neighbor(outgroup, branch_len));
		newRoot->neighbors.push_back(new Neighbor(outgroup->neighbors[0]->node, 0.0));
		outgroup->neighbors[0]->node->replaceNeighbor(outgroup, newRoot, 0.0);
		outgroup->replaceNeighbor(outgroup->neighbors[0]->node, newRoot, branch_len);
	
		//create a fake root as a leaf, connected to the above fake root ==> tree now can be treated as unrooted tree, every node has degree of either 1 or 3. This fake root always has state 1.
		Node* fakeRoot = new Node(leafNum, ROOT_NAME);
		fakeRoot->neighbors.push_back(new Neighbor(newRoot, 0.0));
		newRoot->neighbors.push_back(new Neighbor(fakeRoot, 0.0));
		root = fakeRoot;
	}
}

void Tree::updateRoot(StringVec taxa)
{
	if (rooted == true && !taxa.empty())
		outError(ERR_CONFLICT_ROOT);
	if (strcmp(root->name.c_str(),ROOT_NAME) == 0 )
		outError(ERR_CONFLICT_ROOT);
	//number of taxa in the outgroup
	int outNum = taxa.size();	
	if (!rooted)
	{
		
		Node *otherNode = new Node();
		//user provides outgroup, find the branch (split) on the tree corresponding to those taxa		
		if (outNum > 0)
		{
			//form a split (according to the seqID) reflecting the outgroup set
			Split givenSplit(leafNum);
			for (int i=0; i<outNum; i++)
			{
				int curSeqID = findSeqID(taxa[i]);
				if (curSeqID == -1)
				{
					string err = ERR_NO_LEAF_NAME;
					err += taxa[i];
					outError(err);
				}
				givenSplit.addTaxon(curSeqID);
			}
			//find whether the givenSplit is actually on the tree, return 1 node, the other node of the branch is otherNode;			
			Split *tempSplit = new Split(leafNum);
			Node *newOutGroup = findSplit(givenSplit,tempSplit,otherNode);
			//not found
			if (newOutGroup == NULL)
			{				
				outError("No branch on the tree contains the given taxon set");
			}			
			//found a node with descendants having the given names ==> assign this Node to outgroup
			outgroup = newOutGroup;			
		}
		else
		{
			//user does not provide any outgroup, assign outgroup to the first leaf in Nexus file by default
			outgroup = root;
			otherNode = outgroup->neighbors[0]->node;
		}
		
		//create an internal node as the root of the binary tree, determined by outgroup and otherNode
		Node* newRoot = new Node(nodeNum, INNER_ROOT_NAME);
		double branch_len; //= outgroup->neighbors[0]->length;
		for (NeighborVec::iterator it = outgroup->neighbors.begin(); it != outgroup->neighbors.end(); it++)
			if ((*it) -> node == otherNode) branch_len = (*it)->length;		 
		newRoot->neighbors.push_back(new Neighbor(outgroup, branch_len));
		newRoot->neighbors.push_back(new Neighbor(otherNode, 0.0));
		otherNode->replaceNeighbor(outgroup, newRoot, 0.0);
		outgroup->replaceNeighbor(otherNode, newRoot, branch_len);
	
		//create a fake root as a leaf, connected to the above fake root ==> tree now can be treated as unrooted tree, every node has degree of either 1 or 3. This fake root always has state 1.
		Node* fakeRoot = new Node(leafNum, ROOT_NAME);
		fakeRoot->neighbors.push_back(new Neighbor(newRoot, 0.0));
		newRoot->neighbors.push_back(new Neighbor(fakeRoot, 0.0));
		root = fakeRoot;
	}	
}

Node* Tree::findSplit(const Split querySplit, Split *resp, Node *&retNode2, Node* node, Node *dad)
{
	assert(querySplit.getNTaxa() == leafNum);
	assert(resp->getNTaxa() == leafNum);
	if (!node) 
	{
		node = root;
		if (node->seqID >= 0)
			resp->addTaxon(node->seqID);
		if (*resp == querySplit)
		{			
			retNode2 = node->neighbors[0]->node;			
			return node;
		}	
	}
	bool has_child = false;
	FOR_NEIGHBOR_IT(node, dad, it) {
		Split *sp = new Split(leafNum);
		Node* res = findSplit(querySplit, sp, retNode2, (*it)->node, node);
		if (res)
			return res;
		*resp += *sp;			
		has_child = true;
	}
	if (!has_child)
		resp->addTaxon(node->seqID);
	if (*resp == querySplit)
	{		
		retNode2 = dad;
		return node;
	}
	return NULL; 
}


/**
	\fn updateSeqID
*/
void Tree::updateSeqID(StringVec leafNameVec, Node *node, Node *dad){
	if (node == NULL) node = root;
	if (node->isLeaf())
	{
		//if (node == root && rooted) //is commented out on 2008.11.04
		if (node == root) //added on 2008.11.04
		{
// 			node->seqID = -1; //commented out on 2009.10.14
			if (node->id >= 0 && node->id < leafNum)
			{
				int seq_id = searchString(node->name, leafNameVec);
				if (seq_id == -1)
					outError(ERR_NO_SEQUENCE_NAME, node->name);
				else node->seqID = seq_id;
			}				
		}
		else
		{
			int seq_id = searchString(node->name, leafNameVec);
			if (seq_id == -1)
				outError(ERR_NO_SEQUENCE_NAME, node->name);
			else node->seqID = seq_id;
		}
	}	
	FOR_NEIGHBOR_IT(node, dad, it)
	{
		updateSeqID(leafNameVec, (*it)->node, node);
	}
}


void Tree::setBrId(Node *node, Node *dad)
{
	if (node == NULL) node = root;	
	FOR_NEIGHBOR_IT(node, dad, it)
	{	
		(*it)->brId = brNum;

		for (NeighborVec::iterator nn = (*it)->node->neighbors.begin(); nn != (*it)->node->neighbors.end(); nn++)
		{
			if ((*nn)->node == node)
				(*nn)->brId = brNum;
		}
		brNum++;
		setBrId((*it)->node, node);	
	}
}

void Tree::resetBrId()
{
	brNum = 0;
	setBrId();
}

void Tree::convertSplits(MapBrSplit &brSplit, Split *resp, Node* node, Node *dad)
{
	/*if (!node)
		node = root;
	if (node == root && rooted)
	{
		node = root->neighbors[0]->node;
		dad = root;
	}	
	assert(resp->getNTaxa() == leafNum);
	bool has_child = false;
	FOR_NEIGHBOR_IT(node, dad, it) {
		Split *sp = new Split(leafNum);
		convertSplits(brSplit, sp, (*it)->node, node);
		*resp += *sp;
		brSplit.insert( mBrSplit::value_type((*it)->brId,sp) );
		has_child = true;
	}
	if (!has_child)
		resp->addTaxon(node->seqID);	*/
/*	if (!node || node == root)
	{
		node = root->neighbors[0]->node;
		dad = root;
	}*/
	if (!node)
		node = root;
	assert(resp->getNTaxa() == leafNum);
	bool has_child = false;
	FOR_NEIGHBOR_IT(node, dad, it) {
		Split *sp = new Split(leafNum, (*it)->length);
		convertSplits(brSplit, sp, (*it)->node, node);
		*resp += *sp;
		brSplit.insert( MapBrSplit::value_type((*it)->brId,sp) );
		has_child = true;
	}
	if (!has_child)
		resp->addTaxon(node->seqID);	
}




