/***************************************************************************
 *   Copyright (C) 2010 by Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler   *
 *   minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef SPLIT_H
#define SPLIT_H

#include "utility.h"

/**
	Defines a split (a branch) on the tree by a boolean vector. Taxon order should be identical to that in the alignment.
	@author Minh Anh Thi Nguyen, Tanja Gesell, Arndt von Haeseler <minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at>
*/
class Split: public BoolVec
{
public:
	/**
		empty contructror
	*/
    Split();
	/**
		construction from a given BoolVec
		@param aBoolVec the given vector of booleans
		@param aweight weight of the split, default is 0
	*/
	Split(const BoolVec aBoolVec, double aweight = 0.0);
	/**
		onstructor
		@param antaxa number of taxa
		@param aweight weight of the split, default is 0
	*/
	Split(int antaxa, double aweight = 0.0);
	/**
		constructor copy from another split
		@param sp split to be copied from
	*/
	Split(const Split &sp);

	/**
		construct the split from a taxon list
		@param antaxa number of taxa
		@param taxa_list list of taxa in one side of split
		@param aweight weight of the split, default is 0
	*/
	Split(int antaxa, vector<int> taxa_list, double aweight = 0.0);

	/**
		print infos of the split
		@param out the output stream
	*/
	void printSplit(ostream &out);

	
	/**
			destructor
	*/
    ~Split();

	/**
			get number of taxa
			@return number of taxa
	*/
	inline int getNTaxa() const {
		return ntaxa;
	}
	
	/**
			@return TRUE if the set is empty
	*/
	bool isEmpty();

	/**
		add a taxon into the split
		@param tax_id id of taxon from 0..ntaxa-1
	*/
	void addTaxon(int tax_id);

	/**
		remove a taxon from the split
		@param tax_id id of taxon from 0..ntaxa-1
	*/
	void removeTaxon(int tax_id);


	/**
		set number of taxa
		@param antaxa number of taxa
	*/
	void setNTaxa(int antaxa);

	/**
		check if the split contain a given taxon
		@param taxonID id of the taxon
	*/
	bool containTaxon(const int taxonID);

	 /**
		compare two split, do not compare the weight
		@param sp the target split
		@return TRUE if equal, FALSE otherwise
	*/
	bool operator==(const Split &sp) const;

	/**
		add all taxa from another split into this split (union)
		@param sp a split
	*/
	Split &operator+=(Split &sp);

	/**
		get the intersection with another split
		@param sp a split
	*/
	Split &operator*=(Split &sp);

	/**
		assignment
		@param sp a split
	*/
	Split &operator= (const Split &sp);

	/**
		inline function get weight
		@RETURN weight of the split
	*/
	inline double getWeight() const
	{
		return weight;
	}

	/**
		inline function set weight
		@param aweight: new weight for this split
	*/
	inline void setWeight(const double aweight)
	{
		weight = aweight;
	}

protected:
	/**
		number of taxa
	*/
	int ntaxa;
	/**
		weight of a split (or the corresponding branch length on a tree)
	*/
	double weight;
};

/**
	compare weights of 2 splits
	@RETURN weight of the former is smaller than weight of the latter
*/
inline int splitweightcmp(const Split* a, const Split* b)
{
        return (a->getWeight() < b->getWeight());
}

/**
	Define a map between a branch on the tree (via brId) and the corresponding split
*/
typedef map < int, Split*, std::less< int > > MapBrSplit;

/**
	prepare vector of splitID (or branchID) and their cummutative branch length. This is to 
	select branches according to selection types (PRO, INV_PRO, NONE)
	@param mapBrSplit a map between branchID and the split
	@param seType selection type
		PRO (default) proportional to the branch length
		INV_PRO inversely proportional to the branch length
		NONE return -1 at this time
	@param ret (OUT) return a map of double (the cummutative value) and int (branch ID)
	@return -1 if seType = 0 or not succeed, 0 otherwise.
	NOTE the return map "ret" may have less element than mapBrSplit if there exists branch length of 0.
	This also mean, the branch of length 0 will NOT be considered!
	
*/
int branchCumLen (const MapBrSplit mapBrSplit, DoubleIntMap & ret, const int seType = PRO);
#endif
