/***************************************************************************
 *   Copyright (C) 2010 by Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler   *
 *   minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "split.h"


Split::Split()
				: BoolVec()
{
	ntaxa = 0;
	weight = 0.0;
}

Split::Split(const BoolVec aBoolVec, double aweight)
				: BoolVec(aBoolVec)
{
	ntaxa = aBoolVec.size();
	setWeight(aweight);
}

Split::Split(int antaxa, double aweight)
				: BoolVec()
{
	setNTaxa(antaxa);
	setWeight(aweight);
}

Split::Split(const Split &sp)
				: BoolVec(sp)
{
	ntaxa = sp.ntaxa;
	setWeight(sp.getWeight());
}

Split::Split(int antaxa, vector<int> taxa_list, double aweight)
				: BoolVec()
{	
	setNTaxa(antaxa);
	vector<int>::iterator it;
	for (it = taxa_list.begin(); it != taxa_list.end(); it++)
	{
		int value = *it;
		(*this)[value] = true;		
	}
	setWeight(aweight);
}

void Split::setNTaxa(int antaxa)
{
	ntaxa = antaxa;
	resize(ntaxa,0);
	for (iterator it = begin(); it != end(); it++)
		(*it) = false;
}

void Split::printSplit(ostream &out)
{
	out.precision(10);
	out << std::fixed;
	for (int i=0; i < getNTaxa(); i++)
	{
		if ((*this)[i] == true)
			out << "1";
		else
			out << "0";
	}
	out << "\t" << getWeight();
}
Split::~Split()
{
// 	(*this).clear();
}

bool Split::isEmpty()
{
	for (iterator it = begin(); it != end(); it++)
		if (*it == true) return false;
	return true;
}

void Split::addTaxon(int tax_id)
{
	assert(tax_id >= 0 && tax_id < ntaxa);
	(*this)[tax_id] = true;
}

void Split::removeTaxon(int tax_id)
{
	assert(tax_id >= 0 && tax_id < ntaxa);
	(*this) = false;
}

bool Split::containTaxon(const int taxonID)
{
	assert(0 <= taxonID && taxonID < ntaxa);
	return at(taxonID);
}

bool Split::operator==(const Split &sp) const
{
	if (ntaxa != sp.ntaxa) return false;
	bool comp = true;
	//directly comparison
	for (const_iterator it = begin(), it2 = sp.begin(); it != end(); it++, it2++)
		if ((*it) != (*it2))
			comp = false;
	if (comp == true)
		return comp;
	
	//compare with the inverse
	for (const_iterator it = begin(), it2 = sp.begin(); it != end(); it++, it2++)
		if ( (*it) == (*it2) )
			return false;
	return true;
}

Split &Split::operator+=(Split &sp)
{
	assert(sp.ntaxa == ntaxa);
	iterator it1, it2;
	for (it1 = begin(), it2 = sp.begin(); it1 != end(); it1++, it2++) {
		(*it1) = (*it1) or (*it2);
	}
	return *this;
}

Split &Split::operator*=(Split &sp)
{
	assert(sp.ntaxa == ntaxa);
	iterator it1, it2;
	for (it1 = begin(), it2 = sp.begin(); it1 != end(); it1++, it2++) {
		(*it1) = (*it1) and (*it2);
	}
	return *this;	
}

Split &Split::operator= (const Split &sp)
{
// 	assert(ntaxa == sp.ntaxa);
	BoolVec::operator= (sp);
	ntaxa = sp.getNTaxa();
	return *this;
}

int branchCumLen (const MapBrSplit mapBrSplit, DoubleIntMap & ret, const int seType)
{	
	ret.insert(DoubleIntMap::value_type(0.0,-1));
	if ( seType == NONE ) return -1;	
	/**
		Inversely proportional to branch length. What should we do if a branch has length 0?
			*> assign it by the smallest branch?? --> it will have high probability
			**> OR ignore it --> it will have no chance to be chosen.
		<*> makes sense for real branch but how about the fake branch?
		<**> will ignore the fake branch and real branches of length 0!!!
	AT THIS STAGE: ignore <**>!
	*/
	else if ( seType == INV_PRO )
	{			
		double sumLen = 0;
		for ( MapBrSplit::const_iterator it = mapBrSplit.begin(); it != mapBrSplit.end(); it++ )
		{
			sumLen += it->second->getWeight();
		}
		if ( sumLen == 0 ) return -1;
		double sumInvLen = 0;
		for ( MapBrSplit::const_iterator it = mapBrSplit.begin(); it != mapBrSplit.end(); it++ )
		{
			if ( it->second->getWeight() != 0)
				sumInvLen += sumLen/it->second->getWeight();			
		}
		double cumSum = 0;
		for ( MapBrSplit::const_iterator it = mapBrSplit.begin(); it != mapBrSplit.end(); it++ )
		{
			if ( it->second->getWeight() != 0 )
			{
				cumSum += sumLen/it->second->getWeight();
				ret.insert( DoubleIntMap::value_type( cumSum/sumInvLen, it->first ));
			}
		}	
		return 0;
	}
	else if ( seType == PRO )
	{
		double sumLen = 0;
		for ( MapBrSplit::const_iterator it = mapBrSplit.begin(); it != mapBrSplit.end(); it++ )
		{
			sumLen += it->second->getWeight();
		}
		if ( sumLen == 0 ) return -1;
		double cumSum = 0;
		for ( MapBrSplit::const_iterator it = mapBrSplit.begin(); it != mapBrSplit.end(); it++ )
		{
			cumSum += it->second->getWeight();
			ret.insert( DoubleIntMap::value_type( cumSum/sumLen, it->first ));
		}
		return 0;
	}
	else return -1;
}

