/***************************************************************************
 *   Copyright (C) 2010 by Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler   *
 *   minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef NODE_H
#define NODE_H

/**
describes a node on a tree

	@author Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler 
	<minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at>
*/

#include "utility.h"
using namespace std;

//pre-declaratation of class Node
class Node;


/**
	a Neighbor a node in the tree, a node and a neighbor of it define a branch on the tree
*/
class Neighbor {
public:

	/**
		the other end of the branch
	*/
	Node *node;

	/**
		id of this branches
	*/
	int brId;

	/**
		branch length
	*/
	double length;

	/**
		construction of Neighbor class with a node and length
		@param anode the other end of the branch
		@param alength length of branch
	*/
	Neighbor(Node *anode, double alength) {
		brId = -1;
		node = anode;
		length = alength;		
	}

	Neighbor(Node *anode){
		brId = -1;
		node = anode;
		length = -1;		
	}

	Neighbor(int aBrId, Node *anode, double alength) {
		brId = aBrId;
		node = anode;
		length = alength;		
	}

	Neighbor(int aBrId, Node *anode){
		brId = aBrId;
		node = anode;
		length = -1;		
	}

	/**
		destructor
	*/
	virtual ~Neighbor() {}
};

/*
	some macros to transverse neighbors of a node
*/
#define FOR_NEIGHBOR(mynode, mydad, it) \
	for (it = (mynode)->neighbors.begin(); it != (mynode)->neighbors.end(); it++) \
		if ((*it)->node != (mydad))

#define FOR_NEIGHBOR_IT(mynode, mydad, it) \
	for (NeighborVec::iterator it = (mynode)->neighbors.begin(); it != (mynode)->neighbors.end(); it++) \
		if ((*it)->node != (mydad))

#define FOR_NEIGHBOR_DECLARE(mynode, mydad, it) \
	NeighborVec::iterator it; \
	for (it = (mynode)->neighbors.begin(); it != (mynode)->neighbors.end(); it++) \
		if ((*it)->node != (mydad))

/**
	Vector containing a list of Neighbors
*/
typedef vector<Neighbor*> NeighborVec;

class Node{
public:
/**
		default constructor
	*/
    Node();//{id=-1; seqID=-1; fitchScore=0;};

	/**
		constructor with an id
		@param aid id of this node
	*/
	Node(int aid);

	/**
		constructor with an id and a name (in number)
		@param aid id of this node
		@param aname name of this node
	*/
	Node(int aid, int aname);

	/**
		constructor with an id and a name (in string) 
		@param aid id of this node
		@param aname name of this node
	*/
	Node(int aid, const char *aname);

	/**
		destructor 
	*/
	virtual ~Node();
	
	/**
		@return true of this is a leaf
	*/
	bool isLeaf();

	/**
		@return the number of adjacent nodes
	*/
	int degree();

	/**
		@param node the target node
		@return the iterator to the neighbor that has the node. If not found, return NULL
	*/
	Neighbor *findNeighbor(Node *node);

	/**
		@param node the target node
		@return the iterator to the neighbor that has the node. If not found, return neighbors.end()
	*/
	NeighborVec::iterator findNeighborIt(Node *node);


	/**
		replace a neighbor node by a newnode
		@param node old neighbor node
		@param newnode new neighbor node
		@param newlen new length applied for the corresponding branch
	*/
	void replaceNeighbor(Node* node, Node *newnode, double newlen);

/**------------ATTRIBUTES---------------*/
public:
	/**
		id of this node on the tree
	*/
    int id;
	/**
		name of this node
	*/
	string name;
	/**
		list of neighbors. 1 for leaves, 3 for internal node and 2 for the root in case of binary tree
	*/
	NeighborVec neighbors;
	/**
		id of the corresponding sequence if this node is a leaf. Otherwise, -1.
	*/
	int seqID;
};

#endif
