/***************************************************************************
 *   Copyright (C) 2010 by Minh Anh Thi Nguyen, Tanja Gesell and Arndt von Haeseler   *
 *   minh.anh.nguyen@univie.ac.at, tanja.gesell@univie.ac.at, arndt.von.haeseler@univie.ac.at   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <cstdlib>

#include "utility.h"
#include "alignment.h"
#include "tree.h"
#include "split.h"

using namespace std;
/**
 parse the argument in command line parameter list
 @param argc number of arguments
 @param argv array contains the arguments
 @param params (IN/OUT) to store the arguments
 */
void parseArg(int argc, char* argv[], Params &params, Tree *&tree, Alignment &align);

/**
 interacting with users to feed input
 @param argv the vector of strings containg user's arguments
 @param argc number of strings in the vector
 @param start starting element in the vector to be parsed
 @param first this interaction is the first one (since user start the program)
 @param params (IN/OUT) to store the arguments
 */
void interact(const int start, const int argc, const StringVec argv, Params &params, Tree *&tree, bool first = false);

/**
	mutate a given alignment according to a set of branches, mutation types and site positions. After mutating print the resulting alignment 
	into a file (given parameters in params).
	@param source the source alignment
	@param mapBrSplit a map between branchID and splits
	@param branches a vector of branchID, where extra-muations are put on
	@param mutations a vector of types of mutations
	@param positions a vector of site positions to be mutated
	@param params contains information to form the name of the file storing the resulting alignment
	@param id id for the output file name
*/
void mutateAndPrint(Alignment source, const MapBrSplit mapBrSplit, const vector< int > branches, const vector< int > mutations, const vector< int > positions, Params params, int id = -1);

int main(int argc, char *argv[])
{
	Params params;
	Tree *tree;
	Alignment align;	
	
	/**
		print copy right
	*/
  	printCopyright();
	/**
		parse program's arguments and put into params.
	*/
 	parseArg(argc,argv,params, tree, align);

	
	cout << "converting tree into split set ... " << endl;
	
	//generate split systems from the tree
	Split *initSplit = new Split(tree->leafNum);
	MapBrSplit mapBrSplit;
	tree->convertSplits(mapBrSplit, initSplit);
	cout << "BrID\tSplit\tLength" << endl;
	for ( MapBrSplit::const_iterator iter = mapBrSplit.begin(); iter != mapBrSplit.end(); iter++ )
	{
		cout << iter->first << "\t";
		iter->second->printSplit(cout);
		cout << endl;
	}		
	/**
		Main process
	 */
	deriveInput(params);

	/**
		print user's specification
	*/
 	printUsersSpec(params);		
	if ( strcmp(params.pzFile,"") != 0)
	{
		cout << "Compute the chi-square value for patterns in the original alignment... " << endl;	
		cout << "First, run tree-puzzle to compute pattern likelihood..." << endl;
		char reduceAlign[MAX_FILE_NAME];
		int temp = sprintf(reduceAlign,"%s.reduce",params.uprefix);
		IntVec freq = align.getFreqVec();
		//Alignment mut_align(source.mutateTo(mapBrSplit,branches,mutations,positions));
		align.printReducedAlign(reduceAlign,true);		
		DoubleVec logll;
		//puzzleLogll (reduceAlign, params.treeFile, params.pzFile, reduceAlign, logll);
		puzzleLogll (reduceAlign, params.pzTree, params.pzFile, reduceAlign, logll);
		
		int line = freq.size();
		assert (line == logll.size());
		//print observed frequency and logll
		char outAddFile[MAX_FILE_NAME];
		temp = sprintf(outAddFile,"%s.patInfo",params.uprefix);
		try {
			ofstream outAdd;
			outAdd.exceptions(ios::failbit | ios::badbit);
			outAdd.open(outAddFile);
			//print					
			for (int i = 0; i < line; i++)
				outAdd << align.at(i) << "\t" << freq[i] << "\t" << logll[i] << endl;
			outAdd.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	
		int nsite = align.getNSite();
		//get the observed proportion and change logll to likelihood
		DoubleVec obs;
		for ( int j = 0; j < line; j++ ){
			obs.push_back((double)freq[j]/(double)nsite);
			logll[j] = exp(logll[j]);
		}
		//compute the chi-square value of this alignment according to observed and expected likelihood
		double chisquare = computeChisquare (obs, logll);
		
		//print the chi-square value to a file
		temp = sprintf(outAddFile,"%s.chisq",params.uprefix);
		try {
			ofstream outChi;
			outChi.exceptions(ios::failbit | ios::badbit);
			outChi.open(outAddFile);
			outChi << chisquare << endl;
			outChi.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
		cout << "Finish computing chi-square value for the patterns in the original alignment!" << endl;		
	}

	clock_t beginTime, endTime;

	time_t currentTime;
    time (&currentTime);
    char *date;
    date = ctime(&currentTime);
    cout << "Starting time: " << date << endl;
	beginTime = clock();
	int numBr = params.branches.size();
	int numMut = params.mutations.size();
	int numPos = params.positions.size();
	if ( numBr == params.nExtra && numMut == params.nExtra && numPos == params.nExtra && params.nExtra != 0 ){
		
		mutateAndPrint(align, mapBrSplit, params.branches, params.mutations, params.positions, params);
		if ( params.nRepeat > 1 )
			cout << "Branches, mutation types and site positions are all explicitly given. \nRepetition do not make any sense (same resulting alignments)." << endl;
	}
	else
	{
		if ( params.nExtra <= 0 && !params.usebranch) 
		{
			cout << "None of {-n, -usebranch} is given." << endl;
			return EXIT_SUCCESS;
		}else
		{
			cout << "Seed for random number generator: " << params.seed << endl;
			srand(params.seed);
			for ( int i = 0; i < params.nRepeat; i++ )
			{
				IntVec branches;
				int nExtra = params.nExtra;
				if ( params.usebranch )
				{
					double sumSiteRates = 0;
					if ( !params.pRates.empty() )
					{						
						int tbsNumSites = params.pRates.size();
						for ( int id = 0; id < tbsNumSites; id ++ )
							sumSiteRates += params.pRates[id];							
					}
					if ( sumSiteRates == 0 )
						sumSiteRates = (double)align.getNSite();

					if ( params.nExtra > 0 )
						cout << "Both -n and -usebranch are given, however, we will put extra-substitution following -usebranch! This means -n is not taken into account!" << endl;
					//double alignLen = (double)align.getNSite();
					for ( MapBrSplit::const_iterator it = mapBrSplit.begin(); it != mapBrSplit.end(); it++ )
					{
						double brLen = it->second->getWeight();
						int brEvents = countEvents(brLen,sumSiteRates);
						if (brEvents > 0)
							for ( int eventID = 0; eventID < brEvents; eventID++ )
								branches.push_back(it->first);
					}
					//update number of extra substitutions to be added
					nExtra = branches.size();
				}
				if ( branches.empty())
				{
					//branches are selected randomly according to a uniform distribution
					if ( params.br == 0 )
					{
						if ( params.branches.empty())
						{
							int numBr = mapBrSplit.size();
							selectNumber(1, numBr-1, nExtra, branches, true);
						}
						else
							if ( params.branches.size() != nExtra )
								selectFromIntVec(nExtra, params.branches, branches);
							else
								branches = params.branches;
					}
					else //params.br == 1 
					{
						DoubleIntMap cumLenBrId;
						branchCumLen(mapBrSplit, cumLenBrId, params.br);
						selectIndex(cumLenBrId, nExtra, branches);
						//may also use the Alias method here if the number of branches is large.
					}					
				}

				IntVec mutations; // = params.mutations;
				// to select the types of mutations according to their probabilities we don't use the Alias method.	
				if ( !params.mutations.empty() && !params.mutRates.empty() )
					cout << "A list of substitution types and rates are given. We only take the substitution rates into account. This means the list of substitution types is ignored!" << endl;				
				if ( params.mr != 0 && !params.mutRates.empty() )
				{											
					IntDoubleMap mut_rates;
					for ( int rateID = 1; rateID < MutType; rateID++ )
						mut_rates.insert(IntDoubleMap::value_type(rateID, params.mutRates[rateID-1]));
					DoubleIntMap cumRate_mut;
					mutCumRate (mut_rates, cumRate_mut, params.mr);
					selectIndex(cumRate_mut, nExtra, mutations);
				}else
				{
					if ( params.mutations.empty())
						selectNumber(1,MutType-1, nExtra,mutations,true);
					else
						if ( params.mutations.size() != nExtra )					
							selectFromIntVec(nExtra, params.mutations, mutations);
						else
							mutations = params.mutations;
				}
								
				IntVec positions;// = params.positions; //Need to use the Alias method if the number of categories or alignment length is large.
				if ( !params.positions.empty() && !params.pRates.empty() )
					cout << "A list of site positions and site rates are both given. We only take the site rates into account. This means the list of site positions is ignored!" << endl;
				if ( params.pr != 0 && !params.pRates.empty() ){
					/*
					IntDoubleMap site_rates;
					int tbsNumSites = params.pRates.size();
					for ( int id = 0; id < tbsNumSites; id ++ )
						site_rates.insert(IntDoubleMap::value_type(params.tbsPositions[id], params.pRates[id]));
					DoubleIntMap mutRate_pos;
					mutCumRate(site_rates, cumRate_pos, params.pr);
					*/
					/*This method is very slow if alingment length is large*/
					/*DoubleIntMM site_rates; //DoubleIntMM = double int multimap
					int tbsNumSites = params.pRates.size();
					for ( int id = 0; id < tbsNumSites; id ++ )
						site_rates.insert(DoubleIntMM::value_type(params.pRates[id], params.tbsPositions[id]));
					DoubleIntMap cumRate_pos;
					//cout << "computing mutCumRate ...." << endl;
					mutCumRate2(site_rates, cumRate_pos, params.pr);
					//for ( DoubleIntMap::const_iterator it = cumRate_pos.begin(); it != cumRate_pos.end(); it++ )
					//	cout << it->first << " " << it->second << endl;
					//cout << "selecting indices ... " << endl;
					selectIndex(cumRate_pos, nExtra, positions);
					//for (int test = 0; test < params.nExtra; test ++ )
					//	cout << positions[test] << endl;*/
					
					/**The Alias method*/
					//First, generate the cutoff and alias vectors
					DoubleVec cutoffVec;
					IntVec aliasVec;
					int curTemp = generateCutoffAlias(params.pRates, cutoffVec, aliasVec);					
					if ( curTemp == -1 ) outError("Error occurred while generating cutoff and alias for selecting alignment positions (generateCutoffAlias function)");
					//Second, select nExtra indices
					IntVec indexVec;
					curTemp = selectIndexAlias(cutoffVec, aliasVec, nExtra, indexVec);
					if ( curTemp == -1 ) outError("Error occurred while selecting indices with the Alias method in order to select alignment positions (selectIndexAlias function)");
					for ( int tempID = 0; tempID < nExtra; tempID ++ )
						positions.push_back(params.tbsPositions[indexVec[tempID]]);
					
				}else
				{
					if ( params.positions.empty())
					{
						int nsite = align.getNSite();
						selectNumber(0, nsite-1, nExtra, positions, true);
					} else
						if ( params.positions.size() != nExtra )
							selectFromIntVec(nExtra, params.positions, positions);
						else
							positions = params.positions;					
				}
								
				/*for (int j = 0; j < params.nExtra; j++ )
					cout << branches[j] << "\t" << mutations[j] << "\t" << positions[j] << endl;*/
				mutateAndPrint(align, mapBrSplit, branches, mutations, positions, params, i);
			}
		}
	}
	endTime = clock();
	cout << "\nTime used: " << (double)(endTime - beginTime) / CLOCKS_PER_SEC << " seconds." << endl;
	time (&currentTime);
	date = ctime(&currentTime);
	cout << "Finishing time: " << date << endl;
  	return EXIT_SUCCESS;
}

/**
 parse the argument in command line parameter list	
 */
//void parseArg(int argc, char* argv[], Params &params, Tree *&tree, Alignment *&align)
void parseArg(int argc, char* argv[], Params &params, Tree *&tree, Alignment &align)
{
	params.alignFile = new char[MAX_FILE_NAME];//= NULL;
	params.treeFile = new char[MAX_FILE_NAME];//= NULL;
	params.pzTree = new char[MAX_FILE_NAME];
	params.nExtra = 0;
	params.outgroupFile = new char[MAX_FILE_NAME];
 	strcpy(params.outgroupFile, "");//= NULL;
	params.positionFile = new char[MAX_FILE_NAME];//= NULL;
	params.mutationFile = new char[MAX_FILE_NAME];//= NULL;
	params.branchFile = new char[MAX_FILE_NAME];//= NULL;
	params.bmFile = new char[MAX_FILE_NAME];//= NULL;
	params.bpFile = new char[MAX_FILE_NAME];//= NULL;
	params.bmpFile = new char[MAX_FILE_NAME];//= NULL;
	params.mpFile = new char[MAX_FILE_NAME];//= NULL;
	params.pzFile = new char[MAX_FILE_NAME];//=NULL;
	params.prateFile = new char[MAX_FILE_NAME];
	params.ni = false;
	params.seed = time(0);
	params.mr = 0;
	params.br = 0;
	params.pr = 0;
	params.nRepeat = 1;
	params.uprefix = new char[MAX_FILE_NAME];//= NULL;
	params.sd = false;
	params.add = false;
	params.mutRates = DoubleVec();
	params.pRates = DoubleVec();
	params.ready = false;
	params.outgroupGiven = false;
	params.usebranch = false;
	//parse argv
	if (argc == 1)
	{
 		printHelp();
		exit(0);
	}	
	StringVec argvStr = copyToStringVec(argc,argv);	
	interact (1, argc, argvStr, params, tree, true);
	//finish reading command line argument
	if (strcmp(params.alignFile,"") == 0)
		outError("No alignment file is given. Please use -a!");	
	if (strcmp(params.treeFile,"") == 0)
		outError("No tree file is given. Please use -t!");
	//the alignment
	InputType temp;
	align = Alignment(params.alignFile, temp);
	//the  tree
	tree = new Tree;
	tree->readTree(params.treeFile);
	//update root for the tree if user provide the outgroup
	//if (params.outgroupFile != NULL){
	if (strcmp(params.outgroupFile,"") != 0 ){	
		readStringVec(params.outgroupFile, params.outgroup);
 		tree->updateRoot(params.outgroup);
	}
	//reset branch ID for the tree
	tree->resetBrId();
	//update sequence ID
	tree->updateSeqID(align.getSeqNames());
	//if (tree->getRootName() == ROOT_NAME)
	if ( tree->rooted || params.outgroupGiven )
		cout << "\nYour input (rooted) tree with ID of the branches:" << endl;
	else cout << "\nBelow is your (unrooted) tree. Please root the tree by providing -o before selecting branches!" << endl;
	tree->drawTreeBrId(cout);
	
	/**
	 User does not finish inputting arguments yet
	 */
	if (!params.ready)
	{		
		StringVec instr;
		int count2;
		getInputLine(INTERACTIVE_QUESTION,count2,instr); 				
		interact(0,count2,instr,params, tree, false);
	}
	
	cout << "User finishes inputting!" << endl;
	if (!tree->rooted && !params.outgroupGiven){
		cout << "WARNING: the tree is not rooted by any mean. By default, we rooted it at the first taxon." << endl;
		cout << "The branchIDs are changed and the resulting rooted tree is shown below: " << endl;
		tree->updateRoot("");
		tree->resetBrId();
		tree->drawTreeBrId(cout);
	}
	if ( strcmp(params.pzTree,"") == 0 && strcmp(params.pzFile,"") != 0 )
                strcpy(params.pzTree,params.treeFile);
        cout << "Tree to run puzzle: " << params.pzTree << endl;	
	if (strcmp(params.uprefix,"") == 0)
		params.uprefix = params.alignFile;
	if (params.br < -1 || params.br > 1)
		params.br = 0;
	if (params.mr < -1 || params.mr > 1 )
		params.mr = 0;
}

void interact(const int start, const int argc, const StringVec argv, Params &params, Tree *&tree, bool first)
{
	int count = start;
	try	{
		for (count = start; count < argc; count++)
		{
			if ( strcmp(argv[count].c_str(),"q") == 0 || strcmp(argv[count].c_str(),"Q") == 0 )
				exit(0);
			if ( strcmp(argv[count].c_str(), "-h") == 0 || strcmp(argv[count].c_str(), "--help") == 0)
			{
				if (first){
					printHelp();
					exit(0);
				}else{
					printHelp(PARTLY);					
					StringVec instr;
					int count2;
					getInputLine(INTERACTIVE_QUESTION,count2,instr); 				
					interact(0,count2,instr,params,tree,false);
				}
			}
			if ( strcmp(argv[count].c_str(), "-hh") == 0 )
				printHelp(HIDDEN);
			else if ( strcmp(argv[count].c_str(), "-a") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -a <alignment file name>";
				strcpy(params.alignFile,argv[count].c_str());
			}			
			else if ( strcmp(argv[count].c_str(), "-t") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -t <tree file name>";
				strcpy(params.treeFile, argv[count].c_str());
			}
			 else if ( strcmp(argv[count].c_str(), "-pzTree") == 0)
                        {
                                count++;
                                if (count >= argc)
                                        throw "use -pzTree <tree file name>";
                                strcpy(params.pzTree, argv[count].c_str());
                        }
			else if ( strcmp(argv[count].c_str(), "-p") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -p <position file name>";
				strcpy(params.positionFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-m") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -p <mutation file name>";
				strcpy(params.mutationFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-b") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -b <branch file name>";
				strcpy(params.branchFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-bm") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -bm <branch-mutation file name>";
				strcpy(params.bmFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-bp") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -bp <branch-position file name>";
				strcpy(params.bpFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-mp") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -mp <mutation-position file name>";
				strcpy(params.mpFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-bmp") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -bmp <branch-mutation-position file name>";
				strcpy(params.bmpFile, argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-bprob") == 0) 
			{				
				params.br = 1;
			}
			else if ( strcmp(argv[count].c_str(), "-usebranch") == 0) 
			{				
				params.usebranch = 1;
			}
		/*	else if ( strcmp(argv[count].c_str(), "-br") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -br < Branch selection mode >";
				params.br = convert_int(argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-mr") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -mr < mutation type selection mode >";
				params.mr = convert_int(argv[count].c_str());
			}*/
			else if ( strcmp(argv[count].c_str(), "-mrates") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -mrates <ts tv1 tv2 >";
				params.mutRates.push_back(convert_double(argv[count].c_str()));
				count++;
				if (count >= argc)
					throw "use -mrates <ts tv1 tv2 >";
				params.mutRates.push_back(convert_double(argv[count].c_str()));
				count++;
				if (count >= argc)
					throw "use -mrates <ts tv1 tv2 >";
				params.mutRates.push_back(convert_double(argv[count].c_str()));
				params.mr = 1;			
			}
			/*else if ( strcmp(argv[count].c_str(), "-pr") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -pr < position selection mode >";
				params.pr = convert_int(argv[count].c_str());
			}*/
			else if ( strcmp(argv[count].c_str(), "-prates") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -prates <rates-for-sites filename>";
				strcpy(params.prateFile, argv[count].c_str());
				params.pr = 1;				
			}		
			else if ( strcmp(argv[count].c_str(), "-o") == 0) 
			{
				if (params.outgroupGiven)
					throw "outgroup is already assigned. Restart the program to re-assign outgroup!";
				count++;
				if (count >= argc)
					throw "use -o <outgroup file name>";
				strcpy(params.outgroupFile, argv[count].c_str());
				//update outgroup and show the tree if this is not the first command-line argument feeding
				if (!first){
					readStringVec(params.outgroupFile, params.outgroup);
					//update root for the tree
					tree->updateRoot(params.outgroup);
					//reset branch ID for the tree
					tree->resetBrId();
					cout << "Tree after updating outgroup: " << endl;
					tree->drawTreeBrId(cout);
				}
				params.outgroupGiven = true;
			}
			else if ( strcmp(argv[count].c_str(), "-pz") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -pz < file contains parameters to run puzzle >";
				strcpy(params.pzFile,argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-n") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -n < number of mutations>";
				params.nExtra = convert_int(argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-r") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -r < number of repetition>";
				params.nRepeat = convert_int(argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-seed") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -seed < initialisation for random number generator>";
				params.seed = convert_int(argv[count].c_str());
			}
			else if ( strcmp(argv[count].c_str(), "-prefix") == 0) 
			{
				count++;
				if (count >= argc)
					throw "use -prefix <prefix of output file name>";
				strcpy(params.uprefix, argv[count].c_str());
			}
			else if (strcmp(argv[count].c_str(), "-add") == 0)
				params.add = true;
			else if ( strcmp(argv[count].c_str(), "-sd") == 0 )
				params.sd = true;
			else if ( strcmp(argv[count].c_str(), "-ni") == 0 )
				params.ready = true;
			else if ( strcmp(argv[count].c_str(), "y") == 0 || strcmp(argv[count].c_str(),"Y") == 0 )
				params.ready = true;
			else
			{
				string err = "Unknown argument \"";
				err += argv[count];
				err += "\"";
				throw err;
			}
		}
	} catch (const char* str) {
		outError(str);
	} catch (char* str) {
		outError(str);
	} catch (string str) {
		outError (str);
	} catch (...) {
		string err = "Unknown argument \"";
		err += argv[count];
		err += "\"";
		outError(err);
	}
	if (!first && !params.ready)
	{		
		StringVec instr;
		int count2;
		getInputLine(INTERACTIVE_QUESTION,count2,instr); 				
		interact(0,count2,instr,params, tree, false);		
	}
}

void mutateAndPrint(Alignment source, const MapBrSplit mapBrSplit, const vector< int > branches, const vector< int > mutations, const vector< int > positions, Params params, int id)
{
	char outAlign[MAX_FILE_NAME];
	int temp;
	temp = (id == -1) ? sprintf(outAlign,"%s.mut",params.uprefix) : sprintf(outAlign,"%s.%d.mut",params.uprefix,id);
	Alignment mut_align = source.mutateTo2(mapBrSplit,branches,mutations,positions);
	mut_align.printAlign(outAlign,params.sd);
	
	if (strcmp(params.pzFile,"") != 0 )
	{
		temp = (id == -1) ? sprintf(outAlign,"%s.mut.reduce",params.uprefix) : sprintf(outAlign,"%s.%d.reduce",params.uprefix,id);
		IntVec freq = mut_align.getFreqVec();
		//Alignment mut_align(source.mutateTo(mapBrSplit,branches,mutations,positions));
		mut_align.printReducedAlign(outAlign,true);
		//cout << "TO DO: compute and print pattern freq, pattern likelihood and X^2 value!" << endl;
		DoubleVec logll;
		//puzzleLogll (outAlign, params.treeFile, params.pzFile, params.uprefix, logll);
		//puzzleLogll (outAlign, params.treeFile, params.pzFile, outAlign, logll);
		puzzleLogll (outAlign, params.pzTree, params.pzFile, outAlign, logll);

		int line = freq.size();
		assert (line == logll.size());
		//print observed frequency and logll
		char outAddFile[MAX_FILE_NAME];
		temp = (id == -1) ? sprintf(outAddFile,"%s.mut.patInfo",params.uprefix) : sprintf(outAddFile,"%s.%d.patInfo",params.uprefix,id);
		try {
			ofstream outAdd;
			outAdd.exceptions(ios::failbit | ios::badbit);
			outAdd.open(outAddFile);
			//print					
			for (int i = 0; i < line; i++)
				outAdd << mut_align.at(i) << "\t" << freq[i] << "\t" << logll[i] << endl;
			outAdd.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	
		int nsite = mut_align.getNSite();
		//get the observed proportion and change logll to likelihood
		DoubleVec obs;
		for ( int j = 0; j < line; j++ ){
			obs.push_back((double)freq[j]/(double)nsite);
			logll[j] = exp(logll[j]);
		}
		//compute the chi-square value of this alignment according to observed and expected likelihood
		double chisquare = computeChisquare (obs, logll);
		
		//print the chi-square value to a file
		temp = (id == -1) ? sprintf(outAddFile,"%s.mut.chisq",params.uprefix) : sprintf(outAddFile,"%s.%d.chisq",params.uprefix,id);
		try {
			ofstream outChi;
			outChi.exceptions(ios::failbit | ios::badbit);
			outChi.open(outAddFile);
			outChi << chisquare << endl;
			outChi.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	}
	if (params.add) // print additional information
	{
		char outAddFile[MAX_FILE_NAME];
		temp = (id == -1) ? sprintf(outAddFile,"%s.mut.suppl",params.uprefix) : sprintf(outAddFile,"%s.%d.suppl",params.uprefix,id);
		try {
			ofstream outAdd;
			outAdd.exceptions(ios::failbit | ios::badbit);
			outAdd.open(outAddFile);
			//print
			int line = branches.size();
			outAdd << "BrID\tMut\tSite" << endl;
			for (int i = 0; i < line; i++)
				outAdd << branches[i] << "\t" << mutations[i] << "\t" << positions[i] << endl;
			outAdd.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	}
}

/*void mutateAndPrint(Alignment *source, const MapBrSplit mapBrSplit, const vector< int > branches, const vector< int > mutations, const vector< int > positions, Params params, int id)
{
	char outAlign[MAX_FILE_NAME];
	int temp;
	temp = (id == -1) ? sprintf(outAlign,"%s.mut",params.uprefix) : sprintf(outAlign,"%s.%d.mut",params.uprefix,id);
	Alignment *mut_align = source->mutateTo(mapBrSplit,branches,mutations,positions);
	mut_align->printAlign(outAlign,params.sd);
	
	if (strcmp(params.pzFile,"") != 0 )
	{
		temp = (id == -1) ? sprintf(outAlign,"%s.mut.reduce",params.uprefix) : sprintf(outAlign,"%s.%d.reduce",params.uprefix,id);
		IntVec freq = mut_align->getFreqVec();
		//Alignment mut_align(source.mutateTo(mapBrSplit,branches,mutations,positions));
		mut_align->printReducedAlign(outAlign,true);
		//cout << "TO DO: compute and print pattern freq, pattern likelihood and X^2 value!" << endl;
		DoubleVec logll;
		//puzzleLogll (outAlign, params.treeFile, params.pzFile, params.uprefix, logll);
	//	puzzleLogll (outAlign, params.treeFile, params.pzFile, outAlign, logll);
		puzzleLogll (outAlign, params.pzTree, params.pzFile, outAlign, logll);

		int line = freq.size();
		assert (line == logll.size());
		//print observed frequency and logll
		char outAddFile[MAX_FILE_NAME];
		temp = (id == -1) ? sprintf(outAddFile,"%s.mut.patInfo",params.uprefix) : sprintf(outAddFile,"%s.%d.patInfo",params.uprefix,id);
		try {
			ofstream outAdd;
			outAdd.exceptions(ios::failbit | ios::badbit);
			outAdd.open(outAddFile);
			//print					
			for (int i = 0; i < line; i++)
				outAdd << mut_align->at(i) << "\t" << freq[i] << "\t" << logll[i] << endl;
			outAdd.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	
		int nsite = mut_align->getNSite();
		//get the observed proportion and change logll to likelihood
		DoubleVec obs;
		for ( int j = 0; j < line; j++ ){
			obs.push_back((double)freq[j]/(double)nsite);
			logll[j] = exp(logll[j]);
		}
		//compute the chi-square value of this alignment according to observed and expected likelihood
		double chisquare = computeChisquare (obs, logll);
		
		//print the chi-square value to a file
		temp = (id == -1) ? sprintf(outAddFile,"%s.mut.chisq",params.uprefix) : sprintf(outAddFile,"%s.%d.chisq",params.uprefix,id);
		try {
			ofstream outChi;
			outChi.exceptions(ios::failbit | ios::badbit);
			outChi.open(outAddFile);
			outChi << chisquare << endl;
			outChi.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	}
	if (params.add) // print additional information
	{
		char outAddFile[MAX_FILE_NAME];
		temp = (id == -1) ? sprintf(outAddFile,"%s.mut.suppl",params.uprefix) : sprintf(outAddFile,"%s.%d.suppl",params.uprefix,id);
		try {
			ofstream outAdd;
			outAdd.exceptions(ios::failbit | ios::badbit);
			outAdd.open(outAddFile);
			//print
			int line = branches.size();
			outAdd << "BrID\tMut\tSite" << endl;
			for (int i = 0; i < line; i++)
				outAdd << branches[i] << "\t" << mutations[i] << "\t" << positions[i] << endl;
			outAdd.close();		
		} catch (ios::failure) {
			outError(ERR_WRITE_OUTPUT, outAddFile);
		}
	}
}*/

